
/* --- feature switches --- */

#define _POSIX_SOURCE 1

/* --- system headers --- */

#include <ctype.h>

/* --- local headers --- */

#include "geotouch.h"

/* --- typedefs and function prototypes --- */

/* One record in UW station file files */

typedef struct
  {
  char name[5];  
  float lat1, lat2, lon1,lon2; 
  float elevation;
  char NS, EW;
  } station_rec_uw;

int getline(FILE *input, char *s, int lim);
void gstri(char *string1, int begin, int num, int *i);
void gstrf(char *string1, int begin, int num, float *i);
void gstrf2(char *string1, int begin, int num, float *i, int dec);
void gstrl(char *string1, int begin, int num, long *i);
void gstrc(char *string1, int begin, int num, char *i);

/* --- functions --- */

/* look at first line in station file
    and decide which of four kinds it might be :
3 = UW network
APW   46 39 28.00  122 39 24.30   0.457  12/80  % WR  Alpha Peak
	USNSN:
1 = USNSN 1
ALQ    34.9425N 106.4575W  1849  Albuquerque, New Mexico, USA
	or:

2 = USNSN 2
ALQ    34.9425 -106.4575   1849  Albuquerque, New Mexico, USA

c...|....1....|....2....|....3....|....4....|....5....|....6....|....7.$

0= Original default JML style input
usg3 34N43.0000 116W50.0000 0.000
*/

int Kind_Sta(char *line)
  {
  char t0[10], t1[10], t2[10], t3[10], t4[10], t5[10];
  int i, kind = 0, one = 1, i1, i2, i3 ,i4, i5;

  i = 0;
  gstrc(line, i, one, t0);

  i = 7;
  gstrc(line, i, one, t1);
  i1 = isalpha(t1[0]);

  i = 19;
  gstrc(line, i, one, t2);
  i2 = isalpha(t2[0]);

  i = 14;
  gstrc(line, i, one, t3);
  i3 = isalpha(t3[0]);

  i = 24;
  gstrc(line, i, one, t4);
  i4 = isalpha(t4[0]);

  i = 34;
  gstrc(line, i, one, t5);
  i5 = isalpha(t5[0]);

  if(i1 && i2) kind = 0;
  if(i3 && i4) kind = 2;
  if(!(i1 && i2) && (i3 && i4)) kind = 1;
  if(!(i3 && i4) && !(i1 && i2)) kind = 3;

  if(kind == 3 && i5 > 0) kind = 2;
  return(kind);
  }

bool_t read_jlsta(FILE *f, station_rec_uw *a)
  {
  char line[100], ch[5];
  int i;
  float temp2;

  i = getline(f, line, 100);
  if(i < 1) return(FALSE);

  strncpy(ch, line, 4);
  ch[4] = '\0';
  strcpy(a->name, ch);
  gstrf(line, 5, 2, &a->lat1);
  gstrf(line, 8, 7, &a->lat2);
  gstrc(line, 7, 1, &a->NS);
  gstrf(line, 16, 3, &a->lon1);
  gstrf(line, 20, 7, &a->lon2);
  gstrc(line, 19, 1, &a->EW);
  gstrf(line, 28, 5, &temp2);
  a->elevation = temp2;
  return(TRUE);
  }

bool_t read_uwsta(FILE *f, station_rec_uw *a)
  {
  char line[100], ch[5];
  int i;
  float temp2;

  i = getline(f, line, 100);
  if(i < 1) return(FALSE);

  strncpy(ch, line, 4);
  ch[4] = '\0';
  strcpy(a->name, ch);
  gstrf(line, 5, 2, &a->lat1);
  gstrf(line, 8, 7, &a->lat2);
  gstrc(line, 7, 1, &a->NS);
  gstrf(line, 16, 3, &a->lon1);
  gstrf(line, 20, 7, &a->lon2);
  gstrc(line, 19, 1, &a->EW);
  /* fprintf(stderr, "%s %f %f %f %f\n", a->name, a->lat1, a->lat2, a->lon1,
     a->lon2); */ 
  gstrf(line, 28, 5, &temp2);

  a->elevation = temp2;

  return(TRUE);
  }

/*

	Ours:
APW   46 39 28.00  122 39 24.30   0.457  12/80  % WR  Alpha Peak
c...|....1....|....2....|....3....|....4....|....5....|....6....|....7.$

*/

bool_t read_Malone_sta(FILE *f, station_rec_uw *a)
  {
  char line[100], ch[5];
  int i, i1, i2;
  float f1, f2, temp2;


  i = getline(f, line, 100);
  if(i < 1) return(FALSE);

  strncpy(ch, line, 4);
  ch[4] = '\0';
  strcpy(a->name, ch);

  gstrf(line, 5, 3, &a->lat1);
  gstri(line, 9, 2, &i1);
  gstrf(line, 12, 5, &f1);

  a->lat2 = i1 + f1 / 60.0;
  if(a->lat1 > 0) a->NS = 'N';
  else a->NS = 'S';

  gstrf(line, 18, 4, &a->lon1);
  gstri(line, 23, 2, &i2);
  gstrf(line, 26, 5, &f2);

  a->lon2 = i2 + f2 / 60.0;
  if(a->lon1 > 0) a->EW = 'W';
  else a->EW = 'E';

  gstrf(line, 34, 5, &temp2);
  a->elevation = temp2;

  return(TRUE);
  }

/*

ALQ    34.9425N 106.4575W  1849  Albuquerque, New Mexico, USA
c...|....1....|....2....|....3....|....4....|....5....|....6....|....7.$

*/

bool_t read_USN1_sta(FILE *f, station_rec_uw *a)
  {
  char line[100], ch[5];
  int i;
  float temp2;

  i = getline(f, line, 100);
  if(i < 1) return(FALSE);

  strncpy(ch, line, 4);
  ch[4] = '\0';
  strcpy(a->name, ch);

  gstrf(line, 7, 7, &a->lat1);
  a->lat2 = 0.0;
  gstrc(line, 9, 1, &a->NS);
  gstrf(line, 16, 8, &a->lon1);
  a->lon2 = 0.0;
  gstrc(line, 19, 1, &a->EW);
  gstrf(line, 27, 4, &temp2);
  a->elevation = temp2 / 1000.0;
  return(TRUE);
  }

/*

ALQ    34.9425 -106.4575   1849  Albuquerque, New Mexico, USA
c...|....1....|....2....|....3....|....4....|....5....|....6....|....7.$

*/

bool_t read_USN2_sta(FILE *f, station_rec_uw *a)
{
  char line[100], ch[5];
  int i;
  float temp2;

  i = getline(f, line, 100);
  if (i < 1) return(FALSE);

  strncpy(ch, line, 4);
  ch[4] = '\0';
  strcpy(a->name, ch);
  gstrf(line, 6, 8, &a->lat1);

  a->lat2 = 0.0;
  if(a->lat1 > 0) a->NS = 'N';
  else a->NS = 'S';

  gstrf(line, 15, 9, &a->lon1);
  a->lon2 = 0.0;
  if(a->lon1 > 0) a->EW = 'W';
  else a->EW = 'E';
  gstrf(line, 27, 4, &temp2);
  a->elevation = temp2 / 1000.0;
  return(TRUE);
  }

bool_t GET_A_STA(FILE *inf, station_rec_uw *a, int kind)
  {
  bool_t ans = FALSE;
     
  switch(kind)
    {
    case 0:
      ans = read_uwsta(inf, a);
      break;

    case 1:
      ans = read_USN1_sta(inf, a);
      break;

    case 2:
      ans = read_USN2_sta(inf, a);
      break;

    case 3:
      ans = read_Malone_sta(inf, a);
      break;
    }
  return(ans);
  }

/*

JML format:

usg4 34N44.8000 116W42.0000 0.000
wdc  40N34.8000 122W32.3800 0.300
wood 33N49.8000 116W19.9000 0.122
wplm 33N49.8000 116W19.9000 0.183


     getsta_( str, &a->name, &a->lat1,
	&a->NS,&a->lat2,&a->lon1,&a->EW,&a->lon2,&a->elevation );
      a->name[4] = '\0'; 

	sscanf(str,"%4s %2.0f%c%7.4f %3.0f%c%7.4f  %f\n",
	&a->name, &a->lat1,
	&a->NS,&a->lat2,&a->lon1,&a->EW,&a->lon2,&a->elevation);

  printf("%4s %2.0f%c%7.4f %3.0f%c%7.4f  %f\n",
	a->name, a->lat1,
	a->NS,a->lat2,a->lon1,a->EW,a->lon2,a->elevation);  
           

*/
