/*
 * DGTSL GIVEN A GENERAL TRIDIAGONAL MATRIX AND A RIGHT HAND
 * SIDE WILL FIND THE SOLUTION.
 *
 * ON ENTRY
 *
 * n	INTEGER
 *	IS THE ORDER OF THE TRIDIAGONAL MATRIX.
 *
 * c	float(N)
 *	IS THE SUBDIAGONAL OF THE TRIDIAGONAL MATRIX.
 *	C(2) THROUGH C(N) SHOULD CONTAIN THE SUBDIAGONAL.
 *	ON OUTPUT C IS DESTROYED.
 *
 * d	float(N)
 *	IS THE DIAGONAL OF THE TRIDIAGONAL MATRIX.
 *	ON OUTPUT D IS DESTROYED.
 *
 * e	float(N)
 *	IS THE SUPERDIAGONAL OF THE TRIDIAGONAL MATRIX.
 *	E(1) THROUGH E(N-1) SHOULD CONTAIN THE SUPERDIAGONAL.
 *	ON OUTPUT E IS DESTROYED.
 *
 * b	float(N)
 *	IS THE RIGHT HAND SIDE VECTOR.
 *
 * ON RETURN
 *
 * b	IS THE SOLUTION VECTOR.
 *
 * INFO	INTEGER
 *	= 0 NORMAL VALUE.
 *	= K IF THE K-TH ELEMENT OF THE DIAGONAL BECOMES
 *	EXACTLY ZERO.  THE SUBROUTINE RETURNS WHEN
 *	THIS IS DETECTED.
 */

/* --- feature switches --- */

#define _POSIX_SOURCE 1

/* --- system headers --- */

/* --- local headers --- */

#include "geotouch.h"
#include "jutil.h"

/* --- functions --- */

int jtridiag(int n, float *c, float *d, float *e, float *b)
  {
  int  info = 0, i;
  double d1, d2;
  static int k, kb, kp1, nm1, nm2;
  static double t;

  c[1] = d[1];
  nm1 = n - 1;
  if(!(nm1 < 1))
    {
    d[1] = e[1];
    e[1] = 0.0;
    e[n] = 0.0;

    i = nm1;
    for (k = 1; k <= i; k++)
      {
      kp1 = k + 1;

      /* FIND THE LARGEST OF THE TWO ROWS */

      if(!((d1 = c[kp1], fabs(d1)) < (d2 = c[k], fabs(d2))))
	{
	/* INTERCHANGE ROW */

	t = c[kp1];
	c[kp1] = c[k];
	c[k] = t;
	t = d[kp1];
	d[kp1] = d[k];
	d[k] = t;
	t = e[kp1];
	e[kp1] = e[k];
	e[k] = t;
	t = b[kp1];
	b[kp1] = b[k];
	b[k] = t;
	}

      /* ZERO ELEMENTS */

      if(c[k] == 0.0) return(info = k);

      t = -c[kp1] / c[k];
      c[kp1] = d[kp1] + t * d[k];
      d[kp1] = e[kp1] + t * e[k];
      e[kp1] = 0.;
      b[kp1] += t * b[k];
      }
    }

  if(c[n] == 0.0) return(info = n);

  /* BACK SOLVE */

  nm2 = n - 2;
  b[n] /= c[n];
  if(n == 1) return(info);

  b[nm1] = (b[nm1] - d[nm1] * b[n]) / c[nm1];
  if(nm2 < 1) return(info);

  i = nm2;
  for(kb = 1; kb <= i; kb++)
    {
    k = nm2 - kb + 1;
    b[k] = (b[k] - d[k] * b[k + 1] - e[k] * b[k + 2]) / c[k];
    }
  return(info);
  }

void jspl(float *x, float *y, int n, int ndiv, float *ex, float *ey)
  {
  float *t, *a, *b, *c, *a2, *b2, *c2, *Bx, *By, ttt, tt, tlen,
    *betax, *betay;
  int i, j, k;

  t = alloc_fvec(1, n);
  for(i = 1; i <= n - 1; i++)
    {
    t[i + 1] =
    sqrt((double)((x[i] - x[i - 1]) * (x[i] - x[i - 1])
		  + (y[i] - y[i - 1]) * (y[i] - y[i - 1])));

    }

  a = alloc_fvec(1, n);
  b = alloc_fvec(1, n);
  c = alloc_fvec(1, n);

  a2 = alloc_fvec(1, n);
  b2 = alloc_fvec(1, n);
  c2 = alloc_fvec(1, n);

  Bx = alloc_fvec(1, n);
  By = alloc_fvec(1, n);

  betax = alloc_fvec(1, 4);
  betay = alloc_fvec(1, 4);

  for(i = 2; i <= n - 1; i++)
    {
    a[i] = t[i + 1];
    c[i] = t[i];
    b[i] = 2 * (t[i] + t[i + 1]);

    Bx[i] = (3.0 * (t[i] * t[i] * (x[i] - x[i - 1]) + t[i + 1] * t[i + 1]
		    * (x[i - 1] - x[i - 2])) / (t[i] * t[i + 1]));
    By[i] = (3.0 * (t[i] * t[i] * (y[i] - y[i - 1]) + t[i + 1] * t[i + 1]
		    * (y[i - 1] - y[i - 2])) / (t[i] * t[i + 1]));
    }

  a[n] = 2;
  b[n] = 4;
  b[1] = 1;
  c[1] = 0.5;

  Bx[1] = 1.5 * (x[1] - x[0]) / t[2];
  Bx[n] = 6 * (x[n - 1] - x[n - 2]) / t[n];
  By[1] = 1.5 * (y[1] - y[0]) / t[2];
  By[n] = 6 * (y[n - 1] - y[n - 2]) / t[n];

  /* need to copy input here because routine destroys tridiagonal matrix  */

  for(i = 1; i <= n; i++)
    {
    a2[i] = a[i];
    b2[i] = b[i];
    c2[i] = c[i];
    }

  jtridiag(n, a, b, c, Bx);
  jtridiag(n, a2, b2, c2, By);

  k = 1;
	
  ex[0] = x[0];
  ey[0] = y[0];

  for(i = 1; i < n; i++)
    {
    tt = t[i + 1] * t[i + 1];
    ttt = t[i + 1] * t[i + 1] * t[i + 1];

    betax[1] = x[i - 1];
    betax[2] = Bx[i];
    betax[3] = (3 * (x[i] - x[i - 1]) / tt - 2 * Bx[i] / t[i + 1] - Bx[i + 1]
		/ t[i + 1]);
    betax[4] = (2 * (x[i - 1] - x[i]) / ttt + Bx[i] / tt + Bx[i + 1] / tt);

    betay[1] = y[i - 1];
    betay[2] = By[i];
    betay[3] = (3 * (y[i] - y[i - 1]) / tt - 2 * By[i] / t[i + 1] - By[i + 1]
		/ t[i + 1]);
    betay[4] = (2 * (y[i - 1] - y[i]) / ttt + By[i] / tt + By[i + 1] / tt);

    for(j = 1; j <= ndiv; j++)
      {
      tlen = j * t[i + 1] / ndiv;
      ex[k] = (betax[1] + betax[2] * tlen + betax[3] * tlen * tlen + betax[4]
	       * tlen * tlen * tlen);
      ey[k] = (betay[1] + betay[2] * tlen + betay[3] * tlen * tlen + betay[4]
	       * tlen * tlen * tlen);
      k++;
      }
    }

  free_fvec(t, 1, n);
  free_fvec(a, 1, n);
  free_fvec(b, 1, n);
  free_fvec(c, 1, n);

  free_fvec(a2, 1, n);
  free_fvec(b2, 1, n);
  free_fvec(c2, 1, n);

  free_fvec(Bx, 1, n);
  free_fvec(By, 1, n);

  free_fvec(betax, 1, 4);
  free_fvec(betay, 1, 4);
  }
