
/* --- feature switches --- */

#define _POSIX_SOURCE 1

/* --- system headers --- */

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/* --- local headers --- */

#include "geotouch.h"
#include "xlext.h"
#include "mpic.h"
#include "Gen.h"
#include "xsec.h"
#include "jutil.h"
#include "netcdf.h"

/* --- function prototypes --- */

void citoxyz(int nx, int ny, int nlay, int iway, int *i, int *ix, int *iy,
	     int *iz);
void jgclc(float phio, float lamo, float phi, float lam, float *x, float *y);
void Scale_Image_Global(mpic *m, float max, float min);
void Scale_Image_Set(mpic *m);
int geo2xy(mpic *mplot, float phi, float lam, float *x, float *y);

/* --- typedefs --- */

typedef struct
  {
  int num_dims;		/* number of dimensions 2 = 2D  3 = 3D  */
  int nx;		/* Number of columns */
  int ny;		/* Number of rows */
  int node_offset;	/* 0 for node grids, 1 for pixel grids */
  int  num_layers;	/* number of depth layers  */
  double Lat;		/* Registration Lat coordinate */
  double Lon;		/* Registration Lon coordinate */
  double x_min;		/* Minimum x coordinate */
  double x_max;		/* Maximum x coordinate */
  double y_min;		/* Minimum y coordinate */
  double y_max;		/* Maximum y coordinate */
  double D_min;		/* Minimum Depth coordinate */
  double D_max;		/* Maximum Depth coordinate */
  double *Tops;		/* pointer to array[num_layers+1] of Tops to layers */
  double z_min;		/* Minimum z value */
  double z_max;		/* Maximum z value */
  double x_inc;		/* x increment */
  double y_inc;		/* y increment */
  double z_scale_factor;/* grd values must be multiplied by this */
  double z_add_offset;	/* After scaling, add this */
  char x_units[80];	/* units in x-direction */
  char y_units[80];	/* units in y-direction */
  char D_units[80];	/* units in Depth-direction */
  char z_units[80];	/* grid value units */
  char title[80];	/* name of data set */
  char command[320];	/* name of generating command */
  char remark[160];	/* comments re this data set */
  } GRD_HEADER;

/* GMT: Note on node_offset:
   0 for node grids, 1 for pixel grids
   Assume x_min = y_min = 0 and x_max = y_max = 10 and x_inc = y_inc = 1.
   For a normal node grid we have:
   (1) nx = (x_max - x_min) / x_inc + 1 = 11
   ny = (y_max - y_min) / y_inc + 1 = 11
   (2) node # 0 is at (x,y) = (x_min, y_max) = (0,10) and represents the
   surface
   value in a box with dimensions (1,1) centered on the node.
   For a pixel grid we have:
   (1) nx = (x_max - x_min) / x_inc = 10
   ny = (y_max - y_min) / y_inc = 10
   (2) node # 0 is at (x,y) = (x_min + 0.5*x_inc, y_max - 0.5*y_inc) = (0.5, )
   and represents the surface value in a box with dimensions (1,1)
   centered on the node.
   */

/* --- functions --- */

int read_grd_info_gmt(char *file, GRD_HEADER *header)
  {
  int cdfid, nm[2];
  static long start[2], edge[2];
  double dummy[2];
  char text[480];
  int x_range_id, y_range_id, z_range_id, inc_id, nm_id, z_id;

  start[0] = 0;
  edge[0] = 2;

  /* Open file and get info */

  if(ncopts) ncopts = 0;
  if((cdfid = ncopen(file, NC_NOWRITE)) == -1) return (-1);

  /* Get variable ids */

  x_range_id = ncvarid (cdfid, "x_range");
  y_range_id = ncvarid (cdfid, "y_range");
  z_range_id = ncvarid (cdfid, "z_range");
  inc_id = ncvarid (cdfid, "spacing");
  nm_id	= ncvarid (cdfid, "dimension");
  z_id = ncvarid (cdfid, "z");

  /* Get attributes */
	
  ncattget(cdfid, x_range_id, "units", (void *)header->x_units);
  ncattget(cdfid, y_range_id, "units", (void *)header->y_units);
  ncattget(cdfid, z_range_id, "units", (void *)header->z_units);
  ncattget(cdfid, z_id, "long_name", (void *)header->title);
  ncattget(cdfid, z_id, "scale_factor", (void *) &header->z_scale_factor);
  ncattget(cdfid, z_id, "add_offset", (void *) &header->z_add_offset);
  ncattget(cdfid, z_id, "node_offset", (void *) &header->node_offset);
  ncattget(cdfid, NC_GLOBAL, "title", (void *)header->title);
  ncattget(cdfid, NC_GLOBAL, "source", (void *)text);
  strncpy(header->command, text, 320);
  strncpy(header->remark, &text[320], 160);

  /* Get variables */

  ncvarget(cdfid, x_range_id, start, edge, (void *)dummy);
  header->x_min = dummy[0];
  header->x_max = dummy[1];

  if(header->x_min > 180) header->x_min -= 360;
  if(header->x_max > 180) header->x_max -= 360;

  ncvarget(cdfid, y_range_id, start, edge, (void *)dummy);
  header->y_min = dummy[0];
  header->y_max = dummy[1];
  ncvarget(cdfid, inc_id, start, edge, (void *)dummy);
  header->x_inc = dummy[0];
  header->y_inc = dummy[1];
  ncvarget(cdfid, nm_id, start, edge, (void *)nm);
  header->nx = nm[0];
  header->ny = nm[1];
  ncvarget(cdfid, z_range_id, start, edge, (void *)dummy);
  header->z_min = dummy[0];
  header->z_max = dummy[1];
  ncclose(cdfid);

  return(ncerr);
  }

int read_grd_info_jml(char *file, GRD_HEADER *header)
  {
  int cdfid, nm[3];
  static long start[2], edge[2];
  double dummy[2];
  char text[480];
  int x_range_id, y_range_id, z_range_id, inc_id, nm_id, z_id, D_range_id,
    latlon_id, tops_id;

  start[0] = 0;
  edge[0] = 2;

  /* Open file and get info */

  if(ncopts) ncopts = 0;
  if((cdfid = ncopen (file, NC_NOWRITE)) == -1) return(-1);

  /* Get variable ids */

  x_range_id = ncvarid(cdfid, "x_range");
  y_range_id = ncvarid(cdfid, "y_range");
  z_range_id = ncvarid(cdfid, "z_range");
  D_range_id = ncvarid(cdfid, "D_range");
  inc_id = ncvarid(cdfid, "spacing");
  nm_id	= ncvarid(cdfid, "dimension");
  z_id = ncvarid(cdfid, "z");
  latlon_id = ncvarid(cdfid, "latlon");
  tops_id = ncvarid(cdfid, "tops");

  /* Get attributes */
	
  ncattget(cdfid, x_range_id, "units", (void *)header->x_units);
  ncattget(cdfid, y_range_id, "units", (void *)header->y_units);
  ncattget(cdfid, D_range_id, "units", (void *)header->D_units);
  ncattget(cdfid, z_range_id, "units", (void *)header->z_units);
  ncattget(cdfid, z_id, "long_name", (void *)header->title);
  ncattget(cdfid, z_id, "scale_factor", (void *) &header->z_scale_factor);
  ncattget(cdfid, z_id, "add_offset", (void *) &header->z_add_offset);
  ncattget(cdfid, z_id, "node_offset", (void *) &header->node_offset);
  ncattget(cdfid, NC_GLOBAL, "title", (void *)header->title);
  ncattget(cdfid, NC_GLOBAL, "source", (void *)text);
  strncpy(header->command, text, 320);
  strncpy(header->remark, &text[320], 160);

  /* Get variables */

  ncvarget(cdfid, x_range_id, start, edge, (void *)dummy);
  header->x_min = dummy[0];
  header->x_max = dummy[1];

  ncvarget(cdfid, y_range_id, start, edge, (void *)dummy);
  header->y_min = dummy[0];
  header->y_max = dummy[1];

  ncvarget(cdfid, D_range_id, start, edge, (void *)dummy);
  header->D_min = dummy[0];
  header->D_max = dummy[1];

  ncvarget(cdfid, inc_id, start, edge, (void *)dummy);
  header->x_inc = dummy[0];
  header->y_inc = dummy[1];

  ncvarget(cdfid, latlon_id, start, edge, (void *)dummy);
  header->Lat = dummy[0];
  header->Lon = dummy[1];

  edge[0] = 3;

  ncvarget(cdfid, nm_id, start, edge, (void *)nm);
  header->nx = nm[0];
  header->ny = nm[1];
  header->num_layers = nm[2];

  edge[0] = 2;

  ncvarget(cdfid, z_range_id, start, edge, (void *)dummy);
  header->z_min = dummy[0];
  header->z_max = dummy[1];

  ncclose(cdfid);
  return(ncerr);
  }

int which_kind_grid(char *file)
  {
  int ndims_id, cdfid;
  long  n;
  static long start[2], edge[2];

  start[0] = 0;
  edge[0] = 1;

  if(ncopts) ncopts = 0;

  if((cdfid = ncopen(file, NC_NOWRITE)) == -1)
    {
    fprintf(stderr,"Can't Open the CDF File...trying ascii\n");
    return (-1);
    }

  ndims_id = ncvarid(cdfid, "ndims");
  fprintf(stderr, " ndims_id = %d\n", ndims_id);

  if(ndims_id < 0)
    {
    ncclose(cdfid);
    return(0);
    }

  else
    {
    ncvarget(cdfid, ndims_id, start, edge, &n);
    ncclose(cdfid);
    return(n);
    }
  }

int Get_Image_GMT(char *file, mpic *m)
  {
    GRD_HEADER header;
    int num, nelements, i, k,     cdfid, z_id, *ival,
      nx, ny;
    static long start[2], edge[2];
    float x, y, phio, lamo, phi, lam, *val, *newval;
    
    start[0] = 0;
    edge[0] = 2;
    
    fprintf(stderr, "start getgmt: nx=%d ny=%d nz=%d\n", m->image.nx,
	    m->image.ny, m->image.nz);
    
    if(m->image.nx > 0)
      {
	nelements = m->image.nx * m->image.ny * m->image.nz;
	fprintf(stderr, "nelements=%d\n :  Freeing memory", nelements);
	free_fvec(m->image.val, 0, nelements);
	free_ivec(m->image.col, 0, nelements);
	free_fvec(m->image.z_parts, 0, m->image.nz + 1);
	free_fvec(m->image.Layer_max ,0,m->image.nz);
	free_fvec(m->image.Layer_min,0,m->image.nz);
      }
    
    read_grd_info_gmt(file, &header);
    nx = header.nx;
    ny = header.ny;
    num = nx * ny;
    
    fprintf(stderr, "getgmt: num=%d\n", num);
    if(ncopts) ncopts = 0;
    
    if((cdfid = ncopen(file, NC_NOWRITE)) == -1)
      {
	fprintf(stderr, "Can't Open the File!\n");
	return(-1);
      }
    z_id = ncvarid(cdfid, "z");
    edge[0] = num;
    nelements = num;
    
    /* allocate memory  */
    
    val = alloc_fvec(0, nelements);
    newval = alloc_fvec(0, nelements);
    ival = alloc_ivec(0, nelements);
    
    m->image.val = alloc_fvec(0, nelements);
    m->image.col = alloc_ivec(0, nelements);

    ncvarget(cdfid, z_id, start, edge, (void *)val);
    ncclose(cdfid);
    m->image.nx = nx;
    m->image.ny = ny;
    m->image.nz = 1;
    
    k = 1;
    
    m->image.z_parts = alloc_fvec(0, m->image.nz + 1);
    m->image.Layer_max = alloc_fvec(0, m->image.nz);
    m->image.Layer_min = alloc_fvec(0, m->image.nz);
    m->image.z_parts[0] = 0.0;
    m->image.z_parts[1] = 1.0;
    
    fprintf(stderr," header units= %s\n", header.x_units);
    
    if (!strcmp(header.x_units, "Longitude in degrees"))
      {
	strcpy( m->image.Xunits, "DEG");
      }
    if (!strcmp(header.y_units, "Latitude in degrees"))
      {
	strcpy( m->image.Yunits, "DEG");
      }
    
    fprintf(stderr," Xunits=%s Yunits=%s\n", m->image.Xunits, m->image.Yunits);
    
    printf("getting LATLON values: node_offset=%d header.y_min=%f header.x_min=%f\n",
	   header.node_offset,header.y_min,header.x_min  );
    if(header.node_offset == 0)
      {
	m->image.phi = header.y_min - (header.y_inc / 2.0);
	m->image.lam = header.x_min - (header.x_inc / 2.0);
      }
    else
      {
	m->image.phi = header.y_min;
	m->image.lam = header.x_min;
      }
    
    if(m->image.phi>=0)
      {
	
	phio = m->image.phi;
	lamo = m->image.lam;
	
	phi = m->image.phi + header.y_inc;
	lam = m->image.lam + header.x_inc;
      }
    else
      {
	
	phio = m->image.phi+ny*header.y_inc;
	lamo = m->image.lam;
	
	phi = phio  + header.y_inc;
	lam = lamo + header.x_inc;
	
	
      }
    	m->image.phi = phi ;
	m->image.lam = lam ;
    
    fprintf(stderr, "Origin lat=%f lon=%f\n", phio, lamo); 
    fprintf(stderr, "Corner of GMT file  lat=%f lon=%f\n", phi, lam);

   fprintf(stderr, "%f %f \n", phi, lam);

fprintf(stderr, "header.x_min=%f header.x_max=%f \n", header.x_min, header.x_max);

 fprintf(stderr, "header.y_min=%f header.y_max=%f \n", header.y_min, header.y_max);

 /* by default the data in gmt is stored like this  */
 
    	m->image.phi =  header.y_max ;
	m->image.lam = header.x_min ;

    
    /*  this does not work: geo2xy(m,  phi,  lam, &x, &y); */
    /* this is a potential problem if other projections are used */
    fprintf(stderr," dx=%f dy=%f\n", x, y);
    
    m->image.dx = header.x_inc;
    m->image.dy = header.y_inc;
    
    m->image.spac = m->image.dx;
    m->image.image_skip = -9999999.0;
    
    fprintf(stderr, "Image: dx=%f dy=%f nx=%d ny=%d\n",  m->image.dx,
	    m->image.dy, m->image.nx, m->image.ny);
    
    /* GMT images that come in user_coords are in LAT-LON to make sure
       that they register with the km and pixel grids here we should
       interpolate them onto the km grid */
    
    for(i = 0; i < nelements; i++)
      {
	m->image.val[i] =val[i];
      }
    
    geo2xy(m, m->image.phi, m->image.lam, &m->image.org_minx,
	 &m->image.org_miny);
    m->image.org_maxx = m->image.org_minx+m->image.dx * m->image.nx;
    m->image.org_maxy = m->image.org_miny+m->image.dy * m->image.ny;
    
    Scale_Image_Set(m);
    Scale_Image_Global(m, m->image.Global_image_max, m->image.Global_image_min);
    m->image_flag = 1;
    m->image_layer = 1;
    strcpy(m->image.Type, "GMT");
    
    free_fvec(val, 0, nelements);
    free_fvec(newval, 0, nelements);
    free_ivec(ival, 0, nelements);
    
    fprintf(stderr, " Done reading in GMT Image file...\n");
    
    return(1);
    
  }
/******************************************/
/******************************************/
/***************OLDER version**************/
/******************************************/



int Get_Image_GMT_Old(char *file, mpic *m)
  {
  GRD_HEADER header;
  int num, nelements, i, k, j, ix, iy, iz, iv, iway, cdfid, z_id, *ival,
    nx, ny, kx, ky, kz;
  static long start[2], edge[2];
  float x, y, phio, lamo, phi, lam, *val, *newval;

  start[0] = 0;
  edge[0] = 2;

  fprintf(stderr, "start getgmt: nx=%d ny=%d nz=%d\n", m->image.nx,
	  m->image.ny, m->image.nz);

  if(m->image.nx > 0)
    {
    nelements = m->image.nx * m->image.ny * m->image.nz;
    fprintf(stderr, "nelements=%d\n :  Freeing memory", nelements);
    free_fvec(m->image.val, 0, nelements);
    free_ivec(m->image.col, 0, nelements);
    free_fvec(m->image.z_parts, 0, m->image.nz + 1);
    free_fvec(m->image.Layer_max ,0,m->image.nz);
    free_fvec(m->image.Layer_min,0,m->image.nz);
    }
  
  read_grd_info_gmt(file, &header);
  nx = header.nx;
  ny = header.ny;
  num = nx * ny;

  fprintf(stderr, "getgmt: num=%d\n", num);
  if(ncopts) ncopts = 0;

  if((cdfid = ncopen(file, NC_NOWRITE)) == -1)
    {
    fprintf(stderr, "Can't Open the File!\n");
    return(-1);
    }
  z_id = ncvarid(cdfid, "z");
  edge[0] = num;
  nelements = num;

  /* allocate memory  */

  val = alloc_fvec(0, nelements);
  newval = alloc_fvec(0, nelements);
  ival = alloc_ivec(0, nelements);

  m->image.val = alloc_fvec(0, nelements);
  m->image.col = alloc_ivec(0, nelements);

  ncvarget(cdfid, z_id, start, edge, (void *)val);
  ncclose(cdfid);
  m->image.nx = nx;
  m->image.ny = ny;
  m->image.nz = 1;

  k = 1;

  m->image.z_parts = alloc_fvec(0, m->image.nz + 1);
  m->image.Layer_max = alloc_fvec(0, m->image.nz);
  m->image.Layer_min = alloc_fvec(0, m->image.nz);
  m->image.z_parts[0] = 0.0;
  m->image.z_parts[1] = 1.0;

  fprintf(stderr," header units= %s\n", header.x_units);
  
  /* if (!strcmp(header.x_units, "user_x_unit"))  */

    {

    printf("getting LATLON values: node_offset=%d header.y_min=%f header.x_min=%f\n",
	   header.node_offset,header.y_min,header.x_min  );
    if(header.node_offset == 0)
      {
      m->image.phi = header.y_min - (header.y_inc / 2.0);
      m->image.lam = header.x_min - (header.x_inc / 2.0);
      }
    else
      {
      m->image.phi = header.y_min;
      m->image.lam = header.x_min;
      }

    if(m->image.phi>=0)
      {
    
	phio = m->image.phi;
	lamo = m->image.lam;

	phi = m->image.phi + header.y_inc;
	lam = m->image.lam + header.x_inc;
      }
    else
      {
    
	phio = m->image.phi+ny*header.y_inc;
	lamo = m->image.lam;

	phi = phio  + header.y_inc;
	lam = lamo + header.x_inc;
    
    
      }


     fprintf(stderr, "Origin lat=%f lon=%f\n", phio, lamo); 
     fprintf(stderr, "Corner of GMT file  lat=%f lon=%f\n", phi, lam); 


 
	 jgclc(phio, lamo, phi, lam, &y, &x);
	 

    /*  this does not work: geo2xy(m,  phi,  lam, &x, &y); */
    /* this is a potential problem if other projections are used */
    fprintf(stderr," dx=%f dy=%f\n", x, y);

    m->image.dx = fabs(x);
    m->image.dy = fabs(y);
    }

  m->image.spac = m->image.dx;
  m->image.image_skip = -9999999.0;

  fprintf(stderr, "Image: dx=%f dy=%f nx=%d ny=%d\n",  m->image.dx,
	  m->image.dy, m->image.nx, m->image.ny);

  /* GMT images that come in user_coords are in LAT-LON to make sure
     that they register with the km and pixel grids here we should
     interpolate them onto the km grid */

  /* here i flip the image to go from south to north  */

  for(i = 0; i < nelements; i++)
    {
    ival[i] = 0;
    newval[i] = 0;
    }
  iway = 2;
  fprintf(stderr, "Starting to flip: %f %f %f %f\n", header.x_min,
	  header.x_inc, header.y_min, header.y_inc);
  kz = 1;

  for(i = 0; i < nelements; i++)
    {
    k = i + 1;
    ix = (k % m->image.nx);
    if(ix == 0)	ix = m->image.nx;
    iy = floor((float)(k - ix) / (float)m->image.nx) + 1;
    iz = m->image.ny - iy + 1;
    j = ix + m->image.nx * (iz - 1) - 1;

    /* printf("i=%d k=%d j=%d ix=%d  iy=%d iz=%d\n", i, k, j, ix, iy, iz); */

    lam = header.x_min + (ix - 1) * header.x_inc;
    phi = header.y_max - (iy - 1) * header.y_inc;

    geo2xy(m, phi, lam, &x, &y);
    kx = (int)(x / m->image.dx) + 1;
    ky = (int)(y / m->image.dy) + 1;
    iv = ((ky - 1) * m->image.nx + kx);

    /* if((i%1000)==0)
       {
       fprintf(stderr,"i=%d iv=%d phi=%f lam=%f x=%f y=%f val=%f, kx=%d"
       " ky=%d\n", i, iv, phi, lam, x, y,val[i] , kx, ky);
       }
       */

    /*iv = i;*/

    if(iv < nelements && iv > 0)
      {
      newval[iv] += val[i];
      ival[iv]++;
      }
    }

  for(i = 0; i < nelements; i++)
    {
    if(ival[i] > 0) m->image.val[i] = newval[i] / ival[i];
    else m->image.val[i] = m->image.image_skip;
    }

  geo2xy(m, m->image.phi, m->image.lam, &m->image.org_minx,
	 &m->image.org_miny);
  m->image.org_maxx = m->image.org_minx+m->image.dx * m->image.nx;
  m->image.org_maxy = m->image.org_miny+m->image.dy * m->image.ny;

  Scale_Image_Set(m);
  Scale_Image_Global(m, m->image.Global_image_max, m->image.Global_image_min);
  m->image_flag = 1;
  m->image_layer = 1;
  free_fvec(val, 0, nelements);
  free_fvec(newval, 0, nelements);
  free_ivec(ival, 0, nelements);

  fprintf(stderr, " Done reading in GMT Image file...\n");
  return(1);
  }

int Get_Image_CDF(char *file, mpic *m)
  {
  GRD_HEADER header;
  int num, nelements;
  int             cdfid,tops_id;
  static long     start[2], edge[2];
  int             z_id;
  float           phio, lamo;
  int nx, ny, nz;

  start[0] = 0;
  edge[0] = 2;

  fprintf(stderr, "getgmt: %d %d %d\n", m->image.nx, m->image.ny, m->image.nz);
  if(m->image.nx > 0)
    {
    nelements = m->image.nx * m->image.ny * m->image.nz;
    fprintf(stderr, "nelements=%d\n :  Freeing memory", nelements);
    free_fvec(m->image.val, 0, nelements);
    free_ivec(m->image.col, 0, nelements);
    free_fvec(m->image.z_parts, 0, m->image.nz + 1);
    free_fvec(m->image.Layer_max ,0,m->image.nz);
    free_fvec(m->image.Layer_min,0,m->image.nz);
    }
  read_grd_info_jml(file, &header);
  nx = header.nx;
  ny = header.ny;
  nz = header.num_layers;
  num = nx * ny * nz;
  fprintf(stderr, "getjml: num=%d\n", num);
  if(ncopts) ncopts = 0;

  if((cdfid = ncopen(file, NC_NOWRITE)) == -1)
    {
    fprintf(stderr, "Can't Open the File: %s\n",file);
    return(-1);
    }
  z_id = ncvarid(cdfid, "z");
  edge[0] = num;
  nelements = num;
  m->image.val = alloc_fvec(0, nelements);
  m->image.col = alloc_ivec(0, nelements);

  ncvarget(cdfid, z_id, start, edge, (void *)m->image.val);
  m->image.nx = nx;
  m->image.ny = ny;
  m->image.nz = nz;

  m->image.z_parts = alloc_fvec(0, m->image.nz + 1);
  m->image.Layer_max = alloc_fvec(0, m->image.nz);
  m->image.Layer_min = alloc_fvec(0, m->image.nz);
  tops_id = ncvarid(cdfid, "tops");
  num = header.num_layers + 1;
  edge[0] = num;
  ncvarget(cdfid, tops_id, start, edge, (void *)m->image.z_parts);
  ncclose(cdfid);
  phio = header.Lat;
  lamo = header.Lon;
  
  m->image.phi = phio;
  m->image.lam = lamo;                

  printf("Origin lat=%f lon=%f\n", phio, lamo);

  m->image.dx = header.x_inc;
  m->image.dy = header.y_inc;

  m->image.Global_image_max = header.z_max;
  m->image.Global_image_min = header.z_min;

  m->image.spac = m->image.dx;
  m->image.image_skip = 1000.0;

  geo2xy(m, m->image.phi, m->image.lam, &m->image.org_minx,
	 &m->image.org_miny);
  m->image.org_maxx = m->image.org_minx+m->image.dx*m->image.nx;
  m->image.org_maxy = m->image.org_miny+m->image.dy*m->image.ny;

  Scale_Image_Set(m);
  Scale_Image_Global(m, m->image.Global_image_max, m->image.Global_image_min);
  m->image_flag = 1;
  m->image_layer = 1;
  fprintf(stderr, " Done reading in CDF Image file...\n");

  return (1);
  }

