
/* --- feature switches --- */

#define _POSIX_SOURCE 1

/* --- system headers --- */

/* --- local headers --- */

#include "geotouch.h"
#include "jutil.h"
#include "focproto.h"

/* --- macros --- */

#define DEG2RAD         0.017453293
#define RAD2DEG         57.295778667

/* --- functions --- */

void Dump_Mat(float **r)
  {
  int i, j;

  for(i = 0; i < 3; i++)
    {
    for(j = 0; j < 3; j++) printf("%f ",  r[i][j]);
    printf("\n");
    }
  }

/* ROUTINE CALCULATES X,Y,Z OF A UNIT VECTOR WITH A TREND AND PLUNGE
   OF T AND P. INPUT ANGLES SHOULD BE IN degrees. */

void TP_2_XYZ(float trend, float dip, float *x)
  {
  double t, p;

  t = trend * DEG2RAD;
  p = dip * DEG2RAD;

  x[0] = cos(t) * cos(p);
  x[1] = sin(t) * cos(p);
  x[2] = sin(p);
  }


/* ROUTINE CALCULATES X,Y,Z OF A UNIT VECTOR WITH A TREND AND PLUNGE
   OF T AND P. INPUT ANGLES SHOULD BE IN Radian. */

void rad_TP_2_XYZ(float trend, float dip, float *x)
  {
  double t = trend, p = dip;

  x[0] = cos(t) * cos(p);
  x[1] = sin(t) * cos(p);
  x[2] = sin(p);
  }

void Rotate_vector(float **rmat, float *vec, float *v)
  {
  int i, j;

  for(i = 0; i < 3; i++)
    {
    v[i] = 0.0;

    for(j = 0; j < 3; j++)
      v[i] += rmat[i][j] * vec[j];
    }
  }

void get_roty(float **rotmat, float deg1)
  {

  rotmat[0][0] = cos(deg1);
  rotmat[0][1] = 0.0;
  rotmat[0][2] = -sin(deg1);

  rotmat[1][0] = 0.0;
  rotmat[1][1] = 1.0;
  rotmat[1][2] = 0.0;

  rotmat[2][0] = sin(deg1);
  rotmat[2][1] = 0.0;
  rotmat[2][2] = cos(deg1);
  }

void get_rotx(float **rotmat, float deg1)
  {

  rotmat[0][0] = 1.0;
  rotmat[0][1] = 0.0;
  rotmat[0][2] = 0.0;

  rotmat[1][0] = 0.0;
  rotmat[1][1] = cos(deg1);
  rotmat[1][2] = -sin(deg1);

  rotmat[2][0] = 0.0;
  rotmat[2][1] = sin(deg1);
  rotmat[2][2] = cos(deg1);
  }

/* get a rotation which first rotates about the
   y axis 90 degrees and then roates phi about the X axis
   
   output is a 3X3 matrix which does the rotations of poles */

void vert_rot_mat(float **rotmat, float phi)
  {
  int i, j, k;
  float phirad, deg1, **rotm2, **rotm1;

  rotm2 = alloc_fmat(0, 2, 0, 2);
  rotm1 = alloc_fmat(0, 2, 0, 2);

  /* deg1 = -pi/2   or a 90 degree rotation */

  deg1 = -XMAP_PI_2;
  get_roty(rotm1, deg1);
  phirad = -phi * DEG2RAD;
  get_rotx(rotm2, phirad);

  /* combine these two rotations by matrix multiplication
     rotmat = rotm2 %*% rotm1
     so that the rotm1 is applied first and then  rotm2 rotation */

  for(i = 0; i < 3; i++)
    {
    for(j = 0; j < 3; j++)
      {
      rotmat[i][j] = 0.0;
      for(k = 0; k < 3; k++)
	rotmat[i][j] += rotm2[i][k] * rotm1[k][j];
      }
    }

  free_fmat(rotm2, 0, 2, 0, 2);
  free_fmat(rotm1, 0, 2, 0, 2);
  }

void Rot_TP(float **rotmat, float *strk1, float *dip1)
  {
  float v1[3],  v2[3];

  TP_2_XYZ(*strk1, *dip1, v1);
  Rotate_vector(rotmat,v1,v2);
              
  /* if the vector is pointing down, reflect it through zero
     this is to get the upper hemisphere projection of poles */

  if(v2[2] < 0.0)
    {
    v2[0] *= -1;
    v2[1] *= -1;
    v2[2] *= -1;
    }

  *dip1 = fabs((asin(v2[2])));

  if(v2[1] != 0.0) *strk1 = atan2(v2[1], v2[0]);
  else *strk1 = XMAP_PI_2;

  *dip1 = RAD2DEG * *dip1;
  *strk1 = RAD2DEG * *strk1;
  }

void Rot_foc_phi(float *phi, float *urot, float *udip, float *vrot,
		 float *vdip, float *az1, float *d1, float *az2, float *d2,
		 float *prot, float *pdip, float *trot, float *tdip)
  {
  float **rotmat, p = *phi;

  rotmat = alloc_fmat(0, 2, 0, 2);
  vert_rot_mat(rotmat, p);

  /* Dump_Mat(rotmat); */

  Rot_TP(rotmat, urot, udip);
  Rot_TP(rotmat, vrot, vdip);
 
  *az1 =  *urot - 180.0;
  *d1 = 90.0 - *udip;
  *az2 =  *vrot + 180.0;
  *d2 = 90.0 - *vdip;

  if(*az1 < 0.0) *az1 += 360.0;
  if(*az2 < 0.0) *az2 += 360.0;

  Rot_TP(rotmat, prot, pdip);
  Rot_TP(rotmat, trot, tdip);
  
  free_fmat(rotmat, 0, 2, 0, 2);
  }

/*************************************************************/

/* 
   J.C. Pechmann, July 1986
   converted to C and modified  J.M.Lees Sept. 1994

   Rakcal calculates dip azimuths (dipaz1,dipaz2) and rake angles (rake1,rake2)
   for a focal mechanism from the strike azimuths (az1,az2) & dips (dip1,dip2)
   of the planes.  The strike azimuths must be in degrees measured clockwise
   from north, with the nodal plane dipping down to the right of the strike
   direction.

   Dip angles must be in degrees measured downward from the horizontal. Set 
   dir= +1.0 if the faulting has a reverse component to it and set dir= -1.0
   if the faulting has a normal component to it.  For strike-slip faulting, set
   dir= +1.0 if the plane with the smaller strike azimuth is right lateral.
   All angles returned by this subroutine are in degrees.
  
   Calculate dip azimuths in degrees */

void Rake_Cal(float az1, float dip1, float az2, float dip2, float dir,
	      float *dipaz1, float *rake1, float *dipaz2, float *rake2)
  {
  double rdip1, raz2, rdip2, rdipd1, rdipd2, zslip1, hslip1, yslip1, xslip1,
    zslip2, hslip2, yslip2, xslip2, zstrk1, ystrk1, raz1, xstrk1, zstrk2,
    ystrk2, xstrk2, dot1, dot2;     

  *dipaz1 = az1 + 90.0;
  if(*dipaz1 >= 360.0) *dipaz1 -= 360.0;
  *dipaz2= az2 + 90.0;
  if(*dipaz2 >= 360.0) *dipaz2 -= 360.0;

  /* Convert angles to radians */

  raz1 = az1 * DEG2RAD;
  rdip1 = dip1 * DEG2RAD;
  raz2 = az2 * DEG2RAD;
  rdip2 = dip2 * DEG2RAD;
  rdipd1 = *dipaz1 * DEG2RAD;
  rdipd2 = *dipaz2 * DEG2RAD;

  /* Determine Cartersian coordinates for slip vectors (upper hemisphere) */

  zslip1 = cos(rdip2);
  hslip1 = sin(rdip2);
  yslip1 = hslip1 * cos(rdipd2);
  xslip1 = hslip1 * sin(rdipd2);
  zslip2 = cos(rdip1);
  hslip2 = sin(rdip1);
  yslip2 = hslip2 * cos(rdipd1);
  xslip2 = hslip2 * sin(rdipd1);

  /* Determine Cartesian coordinates for unit vectors in the strike direction
     of each plane */

  zstrk1 = 0.0;
  ystrk1 = cos(raz1);
  xstrk1 = sin(raz1);
  zstrk2 = 0.0;
  ystrk2 = cos(raz2);
  xstrk2 = sin(raz2);

  /* Determine rake angles by taking dot products between slip vectors and
     strike vectors and then taking the inverse cosine */

  dot1 = xslip1 * xstrk1 + yslip1 * ystrk1 + zslip1 * zstrk1;
  dot2 = xslip2 * xstrk2 + yslip2 * ystrk2 + zslip2 * zstrk2;
  if(dot1 > 1.0) dot1 = 1.0;
  if(dot1 < -1.0) dot1 = -1.0;
  if(dot2 > 1.0) dot2 = 1.0;
  if(dot2 < -1.0) dot2 = -1.0;

  *rake1 = acos(dot1) / DEG2RAD;
  *rake2 = acos(dot2) / DEG2RAD;

  /* Adjust rake angles to match the Aki and Richards (p.106) sign convention.
     According to this convention, the rake angle is the angle between the
     direction of movement of the hanging wall and the strike direction of the
     footwall.  The sign is determined by the direction of movement of the
     hanging wall relative to the footwall, where up is defined as being 
     positive.  Thus, if the faulting has a reverse component to it, the rake
     angle is between 0 and 180 degrees.  If the faulting has a normal compo-
     nent to it, the angle is between 0 and -180 degrees.  For strike-slip 
     faulting, the hanging wall is defined as the right-hand block as viewed by
     an observer looking along the strike.  Thus, a left-lateral strike slip 
     fault has a rake angle of 0 and a right-lateral strike-slip fault has a
     rake angle of 180. */

  if(dir < 0) *rake1 -= 180.0;
  if(dir < 0) *rake2 -= 180.0;
  if(*rake1 == -180.0) *rake1 = 180.0;
  if(*rake2 == -180.0) *rake2 = 180.0;
  }

/* determine in a non-ambiguous way which side to shade on a 
   focal sphere

   input angles:
        
   output flag ipol = 1 or 2
           
   */

int Det_polyxy(float strk1, float dip1, float strk2, float dip2,
	       float xt, float yt, float **poly1, float **poly2, int *num1,
	       int *num2)
  {
  double alpha;
  int ipol, ipoly1, ipoly2, ll1, ll2, iflip, ind1, ind2, nn1, nn2, iang = 1;
  float phi, **plan1,**plan2, az1, rdip1, az2, rdip2, t3, p3, angle, tqb, x, y;

  /*  iang = the angle increment drawn along the focal planes and lines */

  /* find intersection; intersection is at cross prodoct of the two poles */
      
  az1 = DEG2RAD * (strk1 - 90);
  rdip1 = DEG2RAD * (dip1);
  az2 = DEG2RAD * (strk2 - 90);
  rdip2 = DEG2RAD * (dip2);

  plan1 = alloc_fmat(0, 200, 0, 1);
  plan2 = alloc_fmat(0, 200, 0, 1);
      
  FM_traj(rdip1, az1, iang, plan1, &ll1);
  FM_traj(rdip2, az2, iang, plan2, &ll2);

  az1 += XMAP_PI_2;
  rdip1 = XMAP_PI_2 - rdip1;
  alpha = rdip2;
  az2 += XMAP_PI_2;
  rdip2 = XMAP_PI_2 - rdip2;

  Cross_prod(az1 * RAD2DEG, rdip1 * RAD2DEG, az2 * RAD2DEG, rdip2 * RAD2DEG,
	     &t3, &p3, &angle);

  t3 = t3 + XMAP_PI_2;
  p3 = XMAP_PI_2 - p3;

  /* plot the null axis */

  phi = p3;
  iflip = 0;
  if(phi > XMAP_PI_2)
    {
    phi = XMAP_PI - phi;
    t3 += XMAP_PI;
    iflip = 1;
    } 

  tqb = XMAP_SQRT2 * sin(phi / 2.0);
  x = tqb * cos(t3);
  y = -tqb * sin(t3);

  ind1 = Find_Near_Ind(plan1, ll1, x, y);
  ind2 = Find_Near_Ind(plan2, ll2, x, y);

  Poly_Gone(plan1, ll1, ind1, plan2, ll2, ind2, x, y, iflip, poly1, &nn1,
	    poly2, &nn2);

  ipoly1 = Inpoly(xt, yt, nn1, poly1);
  ipoly2 = Inpoly(xt, yt, nn2, poly2);
  ipol = 1;
  if(ipoly1 == 1 || ipoly2 == 1) ipol = 0;
  *num1 = nn1;
  *num2 = nn2;
  free_fmat(plan1, 0, 200, 0, 1);
  free_fmat(plan2, 0, 200, 0, 1);
  return(ipol);
  }

/*****************************************************/

/* find the point on either side of the slip vector
   which indicates whether plane is going up or down */

void Slip_Point(float strk1, float dip1, float strk2, float dip2,
		float trot, float tdip, float *plx1r, float *ply1r,
		float *fx, float *fy)
  {
  float angle, rot, ai, d1, strk, az, x1, x2, y1, y2, dis, **foc1, **foc2,
    **poly1, **poly2, pl1 = 0, pl2, ql1 = 0, ql2, xi, pltx, plty, tq, az1, az2,
    rdip1, rdip2, alpha, phi, p3, t3, temx, temy;
  double hypot;
  int ind1, ll1, num1, num2, ipoly1, ipoly2, ipolxy, ipolt, kay, iaorb,
    iang = 1;

  xi = -DEG2RAD * dip2;
  tq = XMAP_SQRT2 * sin(xi / 2.0);
  temx = tq * sin(DEG2RAD * (strk2));
  temy = tq * cos(DEG2RAD * (strk2)); 

  /* now get the NULL axis pole */

  az1 = DEG2RAD * (strk1);
  az2 = DEG2RAD * (strk2);
  rdip1 = DEG2RAD * (dip1);
  rdip2 = DEG2RAD * (dip2);
  
  az1 += XMAP_PI_2;
  rdip1 = XMAP_PI_2 - rdip1;
  alpha = rdip2 ;          
  az2 += XMAP_PI_2;
  rdip2 = XMAP_PI_2 - rdip2;

  Cross_prod(az1 * RAD2DEG, rdip1 * RAD2DEG, az2 * RAD2DEG, rdip2 * RAD2DEG,
	     &t3, &p3, &angle);

  phi = p3;
  rot = RAD2DEG * t3;
  ai =  RAD2DEG * phi;
  t3 += XMAP_PI_2;
  p3 = XMAP_PI_2 - p3;
  phi = p3; 
  rot = RAD2DEG * t3;
  ai =  RAD2DEG * phi;

  if(phi > XMAP_PI_2)
    {
    phi = XMAP_PI - phi;
    t3 += XMAP_PI;
    } 

  t3 += XMAP_PI;

  rot = RAD2DEG * t3;
  ai =  RAD2DEG * phi;
  kay = (int)(rot / 360.0);
  rot -= kay * 360.0;

  /* round off to nearest 10-th of a degree */

  rot = (int)(10 * rot + 0.5) / 10.0;
  ai = (int)(10 * ai + 0.5) / 10.0;

  strk = rot - 90.0;
  az = DEG2RAD * strk;
  d1 = DEG2RAD * ai;
  iang = 1;
            
  /*  using the null axis pole, get the trajectory perpendicular to
      the slip vector */

  foc1 = alloc_fmat(0, 200, 0, 1);
  foc2 = alloc_fmat(0, 200, 0, 1);
  poly1 = alloc_fmat(0, 1000, 0, 1);
  poly2 = alloc_fmat(0, 1000, 0, 1);

  FM_traj(d1, az, iang, foc1, &ll1);
  ind1 = Find_Near_Ind(foc1, ll1, temx, temy);

  if(ind1 <= 0) ind1 = 1;
  if(ind1 >= ll1 - 1) ind1 = ll1 - 2;

  x1 = foc1[ind1 - 1][0];
  x2 = foc1[ind1 + 1][0];
  y1 = foc1[ind1 - 1][1];
  y2 = foc1[ind1 + 1][1];

  dis =  1.7 / 30.0;

  hypot = (x1 - temx) * (x1 - temx) + (y1 - temy) * (y1 - temy);
  if(hypot > 0.0)
    {
    pl1 =   temx + dis * (x1 - temx) / sqrt(hypot);
    ql1 =   temy + dis * (y1 - temy) / sqrt(hypot);
    }

  hypot = (x2 - temx) * (x2 - temx) + (y2 - temy) * (y2 - temy);
  if(hypot > 0.0)
    {
    pl2 = temx + dis * (x2 - temx) / sqrt(hypot);
    ql2 = temy + dis * (y2 - temy) / sqrt(hypot);
    }

  xi = DEG2RAD * (90 - tdip);
  tq = XMAP_SQRT2 * sin(xi / 2.0);
  pltx = tq * sin(DEG2RAD * trot);
  plty = tq * cos(DEG2RAD * trot); 

  ipolt = Det_polyxy(strk1, dip1, strk2, dip2, pltx, plty, poly1, poly2,
		     &num1, &num2);

  ipoly1 =  Inpoly(pl1, ql1, num1, poly1);
  ipoly2 =  Inpoly(pl1, ql1, num2, poly2);

  ipolxy = 1;
  if(ipoly1 == 1 || ipoly2 == 1) ipolxy = 0;
  if(ipolt == ipolxy)
    {
    iaorb = 1;
    hypot = (x1 - temx) * (x1 - temx) + (y1 - temy) * (y1 - temy);
    *fx = (x1 - temx) / sqrt(hypot);
    *fy = (y1 - temy) / sqrt(hypot);
    }
  else
    {
    hypot = (x2 - temx) * (x2 - temx) + (y2 - temy) * (y2 - temy);
    *fx = (x2 - temx) / sqrt(hypot);
    *fy = (y2 - temy) / sqrt(hypot);
    }
  *plx1r = temx;
  *ply1r = temy;

  free_fmat(foc1, 0, 200, 0, 1);
  free_fmat(foc2, 0, 200, 0, 1);
  free_fmat(poly1, 0, 1000, 0, 1);
  free_fmat(poly2, 0, 1000, 0, 1);
  }
