
/**********************************************************************
	    C Implementation of Wu's Color Quantizer (v. 2)
	    (see Graphics Gems CQ_Vol. II, pp. 126-133)

Author:	Xiaolin Wu
	Dept. of Computer Science
	Univ. of Western Ontario
	London, Ontario N6A 5B7
	wu@csd.uwo.ca
	http://www.csd.uwo.ca/faculty/wu/

Algorithm: Greedy orthogonal bipartition of RGB space for variance
	   minimization aided by inclusion-exclusion tricks.
	   For speed no nearest neighbor search is done. Slightly
	   better performance can be expected by more sophisticated
	   but more expensive versions.

The author thanks Tom Lane at Tom_Lane@G.GP.CS.CMU.EDU for much of
additional documentation and a cure to a previous bug.

Free to distribute, comments and suggestions are appreciated.

Code beautified, converted to ANSI-C, de-globalized, and converted into
a library by Mark A. Lindner (markl@gnu.ai.mit.edu).

**********************************************************************/	

/* --- feature switches --- */

#define _POSIX_SOURCE 1

/* --- system headers --- */

#include <stddef.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

/* --- local headers --- */

#include "colorquant.h"

/* --- macros --- */

#define	CQ_RED		  2
#define	CQ_GREEN	  1
#define CQ_BLUE		  0

/* --- typedefs and global variables --- */

struct cq_box
  {
  int r0;			 /* min value, exclusive */
  int r1;			 /* max value, inclusive */
  int g0;  
  int g1;  
  int b0;  
  int b1;
  int vol;
  };

static float m2[33][33][33];
static long int wt[33][33][33], mr[33][33][33], mg[33][33][33], mb[33][33][33];

/* --- functions --- */
 static void CQ_Hist3D(unsigned int *, unsigned int *, unsigned int *,
			unsigned int, unsigned int *,
			long *, long *, long *, long *, float *);
  static void CQ_Moment3D(long *, long *, long *, long *, float *);
  static int CQ_Cut(struct cq_box *, struct cq_box *);
  static float CQ_Var(struct cq_box *);
  static void CQ_Mark(struct cq_box *, int, unsigned int *);
 /*  static long int CQ_Vol(struct cq_box *, long [][][]); */
/*
 *----------------------------------------------------------------------------
 *  CQ_Vol(): compute sum over a box of any given statistic
 *----------------------------------------------------------------------------
 */

static long int CQ_Vol(struct cq_box *cube, long mmt[33][33][33]) 
  {
  return(mmt[cube->r1][cube->g1][cube->b1] 
	 - mmt[cube->r1][cube->g1][cube->b0]
	 - mmt[cube->r1][cube->g0][cube->b1]
	 + mmt[cube->r1][cube->g0][cube->b0]
	 - mmt[cube->r0][cube->g1][cube->b1]
	 + mmt[cube->r0][cube->g1][cube->b0]
	 + mmt[cube->r0][cube->g0][cube->b1]
	 - mmt[cube->r0][cube->g0][cube->b0]);
  }




/*
 *----------------------------------------------------------------------------
 *  CQ_Quantize(): the fast optimal color quantization library interface
 *  routine
 *----------------------------------------------------------------------------
 */

int CQ_Quantize(unsigned int sx,	/* image width (parameter)	    */
		unsigned int sy,	/* image height (parameter)	    */
		unsigned int *K,	/* number of colors (param/result)  */
		qcolor_t **colors,	/* color table (result)		    */
		unsigned int *R,	/* array of pixel reds (param)	    */
		unsigned int *G,	/* array of pixel greens (param)    */
		unsigned int *B,	/* array of pixel blues (param)	    */
		unsigned int **pixels	/* pixel translation table (result) */
		)
  {
  struct cq_box *cube;
  int next;
  register long int i, weight;
  register int k;
  float *vv, temp;
  unsigned int *tag, *ip, size = sx * sy;

 
  if(!(*colors = (qcolor_t *)calloc(*K, sizeof(qcolor_t))))
    {
    return(0);
    }

  if(!(cube = (struct cq_box *)calloc(*K, sizeof(struct cq_box))))
    {
    free((void *)*colors);
    return(0);
    }
  
  if(!(vv = (float *)calloc(*K, sizeof(float))))
    {
    free((void *)*colors);
    free((void *)cube);
    return(0);
    }

  if(!(tag = (unsigned int *)calloc((33 * 33 * 33), sizeof(unsigned int))))
    {
    free((void *)*colors);
    free((void *)cube);
    free((void *)vv);
    return(0);
    }

  if(!(*pixels = (unsigned int *)calloc(size, sizeof(unsigned int))))
    {
    free((void *)*colors);
    free((void *)cube);
    free((void *)vv);
    free((void *)*pixels);
    return(0);
    }

  memset((void *)m2, 0, (33 * 33 * 33) * sizeof(float));
  memset((void *)wt, 0, (33 * 33 * 33) * sizeof(long));
  memset((void *)mr, 0, (33 * 33 * 33) * sizeof(long));
  memset((void *)mg, 0, (33 * 33 * 33) * sizeof(long));
  memset((void *)mb, 0, (33 * 33 * 33) * sizeof(long));

  CQ_Hist3D(R, G, B, size, *pixels, (long *)&(wt[0][0][0]),
	    (long *)&(mr[0][0][0]), (long *)&(mg[0][0][0]),
	    (long *)&(mb[0][0][0]), (float *)&(m2[0][0][0]));

  CQ_Moment3D((long *)&(wt[0][0][0]), (long *)&(mr[0][0][0]),
	      (long *)&(mg[0][0][0]), (long *)&(mb[0][0][0]),
	      (float *)&(m2[0][0][0]));

  cube[0].r0 = cube[0].g0 = cube[0].b0 = 0;
  cube[0].r1 = cube[0].g1 = cube[0].b1 = 32;

  for(next = 0, i = 1; i < *K; i++)
    {
    if(CQ_Cut(&cube[next], &cube[i]))
      {
      /* CQ_Volume test ensures we won't try to CQ_Cut one-cell box */

      vv[next] = ((cube[next].vol > 1) ? CQ_Var(&cube[next]) : 0.0);
      vv[i] = ((cube[i].vol > 1) ? CQ_Var(&cube[i]) : 0.0);
      }
    else
      {
      vv[next] = 0.0;   /* don't try to split this box again */
      i--;              /* didn't create box i */
      }
    next = 0, temp = vv[0];
    for(k = 1; k <= i; k++)
      if(vv[k] > temp)
	temp = vv[k], next = k;

    if(temp <= 0.0)
      {
      /* *K = i + 1; IS this a bug?  why is K changing?*/
      *K = i + 1;
      break;
      }
    }

  for(k = 0; k < *K; k++)
    {
    CQ_Mark(&cube[k], k, tag);
    if((weight = CQ_Vol(&cube[k], wt)))
      {
      (*colors)[k].r = CQ_Vol(&cube[k], mr) / weight;
      (*colors)[k].g = CQ_Vol(&cube[k], mg) / weight;
      (*colors)[k].b = CQ_Vol(&cube[k], mb) / weight;
      }
    else
      {
      /* bogus box k */
      (*colors)[k].r = (*colors)[k].g = (*colors)[k].b = 0;		
      }
    }

  for(i = 0, ip = *pixels; i < size; i++, ip++) *ip = tag[*ip];

  free((void *)cube);
  free((void *)vv);
  free((void *)tag);

  return(1);
  }

/*
 *----------------------------------------------------------------------------
 *  CQ_Hist3D(): build a 3-D color histogram of counts, r/g/b, c^2;
 *  Histogram is in elements 1..HISTSIZE along each axis, element 0 is for
 *  base or marginal value
 *  NB: these must start out 0!
 *  At conclusion of the histogram step, we can interpret
 *   wt[r][g][b] = sum over voxel of P(c)
 *   mr[r][g][b] = sum over voxel of r*P(c)  ,  similarly for mg, mb
 *   m2[r][g][b] = sum over voxel of c^2*P(c)
 *  Actually each of these should be divided by 'size' to give the usual
 *  interpretation of P() as ranging from 0 to 1, but we needn't do that here.
 *----------------------------------------------------------------------------
 */

static void CQ_Hist3D(unsigned int *R, unsigned int *G, unsigned int *B,
		      unsigned int size, unsigned int *pixels,
		      long *vwt, long *vmr, long *vmg, long *vmb, float *m2) 
  {
  register int ind, r, g, b;
  int inr, ing, inb, table[CQ_MAXCOLOR];
  register long int i;

  for(i = 0; i < CQ_MAXCOLOR; i++) table[i] = i * i;

  for(i = 0; i < size; i++)
    {
    r = R[i], g = G[i], b = B[i];
    inr = (r >> 3) + 1, ing = (g >> 3) + 1, inb = (b >> 3) + 1; 

    pixels[i] = ind = (inr << 10) + (inr << 6) + inr + (ing << 5) + ing + inb;

    /*[inr][ing][inb]*/

    vwt[ind]++;
    vmr[ind] += r;
    vmg[ind] += g;
    vmb[ind] += b;
    m2[ind] += (float)(table[r] + table[g] + table[b]);
    }
  }

/*
 *----------------------------------------------------------------------------
 *  CQ_Moment3D(): computer cumulative moments so that we can rapidly
 *  calculate sums over any desired box
 *----------------------------------------------------------------------------
 */

static void CQ_Moment3D(long *vwt, long *vmr, long *vmg, long *vmb, float *m2)
  {
  register unsigned int ind1, ind2;
  register unsigned int i, r, g, b;
  long int line, line_r, line_g, line_b, area[33],
    area_r[33], area_g[33], area_b[33];
  float line2, area2[33];

  for(r = 1; r < 33; r++)
    {
    for(i = 0; i < 33; i++) 
      area2[i] = area[i] = area_r[i] = area_g[i] = area_b[i] = 0;
    for(g = 1; g < 33; g++)
      {
      line2 = line = line_r = line_g = line_b = 0;
      for(b = 1; b < 33; b++)
	{
	ind1 = (r << 10) + (r << 6) + r + (g << 5) + g + b; /* [r][g][b] */
	line += vwt[ind1];
	line_r += vmr[ind1]; 
	line_g += vmg[ind1]; 
	line_b += vmb[ind1];
	line2 += m2[ind1];
	area[b] += line;
	area_r[b] += line_r;
	area_g[b] += line_g;
	area_b[b] += line_b;
	area2[b] += line2;
	ind2 = ind1 - 1089; /* [r-1][g][b] */
	vwt[ind1] = vwt[ind2] + area[b];
	vmr[ind1] = vmr[ind2] + area_r[b];
	vmg[ind1] = vmg[ind2] + area_g[b];
	vmb[ind1] = vmb[ind2] + area_b[b];
	m2[ind1] = m2[ind2] + area2[b];
	}
      }
    }
  }


/*
 *----------------------------------------------------------------------------
 *  The next two routines allow a slightly more efficient calculation
 *  of CQ_Vol() for a proposed subbox of a given box.  The sum of CQ_Top()
 *  and CQ_Bottom() is the CQ_Vol() of a subbox split in the given direction
 *  and with the specified new upper bound.
 *----------------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------------
 *  CQ_Bottom(): compute part of CQ_Vol(cube, mmt) that doesn't depend on
 *  r1, g1, or b1 (depending on dir)
 *----------------------------------------------------------------------------
 */

static long int CQ_Bottom(struct cq_box *cube, unsigned char dir,
			  long mmt[33][33][33])
  {
  switch(dir)
    {
    case CQ_RED:
      return(- mmt[cube->r0][cube->g1][cube->b1]
	     + mmt[cube->r0][cube->g1][cube->b0]
	     + mmt[cube->r0][cube->g0][cube->b1]
	     - mmt[cube->r0][cube->g0][cube->b0]);

    case CQ_GREEN:
      return(- mmt[cube->r1][cube->g0][cube->b1]
	     + mmt[cube->r1][cube->g0][cube->b0]
	     + mmt[cube->r0][cube->g0][cube->b1]
	     - mmt[cube->r0][cube->g0][cube->b0]);

    case CQ_BLUE:
      return(- mmt[cube->r1][cube->g1][cube->b0]
	     + mmt[cube->r1][cube->g0][cube->b0]
	     + mmt[cube->r0][cube->g1][cube->b0]
	     - mmt[cube->r0][cube->g0][cube->b0]);
    }

  return(0);
  }

/*
 *----------------------------------------------------------------------------
 *  CQ_Top(): compute remainder of CQ_Vol(cube, mmt), substituting pos for
 *  r1, g1, or b1 (depending on dir)
 *----------------------------------------------------------------------------
 */

static long int CQ_Top(struct cq_box *cube, unsigned char dir, int pos,
		       long mmt[33][33][33])
  {
  switch(dir)
    {
    case CQ_RED:
      return(mmt[pos][cube->g1][cube->b1] 
	     - mmt[pos][cube->g1][cube->b0]
	     - mmt[pos][cube->g0][cube->b1]
	     + mmt[pos][cube->g0][cube->b0]);

    case CQ_GREEN:
      return(mmt[cube->r1][pos][cube->b1] 
	     - mmt[cube->r1][pos][cube->b0]
	     - mmt[cube->r0][pos][cube->b1]
	     + mmt[cube->r0][pos][cube->b0]);

    case CQ_BLUE:
      return(mmt[cube->r1][cube->g1][pos]
	     - mmt[cube->r1][cube->g0][pos]
	     - mmt[cube->r0][cube->g1][pos]
	     + mmt[cube->r0][cube->g0][pos]);
    }

  return(0);
  }

/*
 *----------------------------------------------------------------------------
 *  CQ_Var(): Compute the weighted variance of a box
 *  NB: as with the raw statistics, this is really the variance * size
 *----------------------------------------------------------------------------
 */

static float CQ_Var(struct cq_box *cube)
  {
  float dr, dg, db, xx;

  dr = CQ_Vol(cube, mr), dg = CQ_Vol(cube, mg), db = CQ_Vol(cube, mb);
  xx = (m2[cube->r1][cube->g1][cube->b1] 
	- m2[cube->r1][cube->g1][cube->b0]
	- m2[cube->r1][cube->g0][cube->b1]
	+ m2[cube->r1][cube->g0][cube->b0]
	- m2[cube->r0][cube->g1][cube->b1]
	+ m2[cube->r0][cube->g1][cube->b0]
	+ m2[cube->r0][cube->g0][cube->b1]
	- m2[cube->r0][cube->g0][cube->b0]);

  return(xx - (dr * dr + dg * dg + db * db) / (float)CQ_Vol(cube, wt));
  }

/*
 *----------------------------------------------------------------------------
 *  CQ_Maximize(): We want to minimize the sum of the variances of two
 *  subboxes. The sum(c^2) terms can be ignored since their sum over both
 *  subboxes is the same (the sum for the whole box) no matter where we split.
 *  The remaining terms have a minus sign in the variance formula,
 *  so we drop the minus sign and maximize the sum of the two terms.
 *----------------------------------------------------------------------------
 */

static float CQ_Maximize(struct cq_box *cube, unsigned char dir, int first,
			 int last, int *cut, long whole_r, long whole_g,
			 long whole_b, long whole_w)
  {
  register long int half_r, half_g, half_b, half_w;
  long int base_r, base_g, base_b, base_w;
  register int i;
  register float temp, max;

  base_r = CQ_Bottom(cube, dir, mr);
  base_g = CQ_Bottom(cube, dir, mg);
  base_b = CQ_Bottom(cube, dir, mb);
  base_w = CQ_Bottom(cube, dir, wt);
  max = 0.0;
  *cut = -1;
  for(i = first; i < last; i++)
    {
    half_r = base_r + CQ_Top(cube, dir, i, mr);
    half_g = base_g + CQ_Top(cube, dir, i, mg);
    half_b = base_b + CQ_Top(cube, dir, i, mb);
    half_w = base_w + CQ_Top(cube, dir, i, wt);

    /* now half_x is sum over lower half of box, if split at i */

    if(half_w == 0) /* subbox could be empty of pixels! */
      continue; /* never split into an empty box */

    else
      temp = (((float)(half_r * half_r) + (float)(half_g * half_g)
	       + (float)(half_b * half_b)) / half_w);

    half_r = whole_r - half_r;
    half_g = whole_g - half_g;
    half_b = whole_b - half_b;
    half_w = whole_w - half_w;

    if(half_w == 0) /* subbox could be empty of pixels! */
      continue;             /* never split into an empty box */

    else
      temp += (((float)(half_r * half_r) + (float)(half_g * half_g)
		+ (float)(half_b * half_b)) / half_w);

    if (temp > max) max = temp, *cut = i;
    }
  return(max);
  }

/*
 *----------------------------------------------------------------------------
 *  CQ_Cut():
 *----------------------------------------------------------------------------
 */

static int CQ_Cut(struct cq_box *set1, struct cq_box *set2)
  {
  unsigned char dir;
  int cutr, cutg, cutb;
  float maxr, maxg, maxb;
  long int whole_r, whole_g, whole_b, whole_w;

  whole_r = CQ_Vol(set1, mr);
  whole_g = CQ_Vol(set1, mg);
  whole_b = CQ_Vol(set1, mb);
  whole_w = CQ_Vol(set1, wt);

  maxr = CQ_Maximize(set1, CQ_RED, set1->r0+1, set1->r1, &cutr,
		  whole_r, whole_g, whole_b, whole_w);
  maxg = CQ_Maximize(set1, CQ_GREEN, set1->g0+1, set1->g1, &cutg,
		  whole_r, whole_g, whole_b, whole_w);
  maxb = CQ_Maximize(set1, CQ_BLUE, set1->b0+1, set1->b1, &cutb,
		  whole_r, whole_g, whole_b, whole_w);

  if((maxr >= maxg) && (maxr >= maxb))
    {
    dir = CQ_RED;
    if (cutr < 0) return(0); /* can't split the box */
    }
  else if((maxg >= maxr) && (maxg >= maxb)) 
    dir = CQ_GREEN;
  else
    dir = CQ_BLUE; 

  set2->r1 = set1->r1;
  set2->g1 = set1->g1;
  set2->b1 = set1->b1;

  switch(dir)
    {
    case CQ_RED:
      set2->r0 = set1->r1 = cutr;
      set2->g0 = set1->g0;
      set2->b0 = set1->b0;
      break;
    case CQ_GREEN:
      set2->g0 = set1->g1 = cutg;
      set2->r0 = set1->r0;
      set2->b0 = set1->b0;
      break;
    case CQ_BLUE:
      set2->b0 = set1->b1 = cutb;
      set2->r0 = set1->r0;
      set2->g0 = set1->g0;
      break;
    }
  set1->vol = ((set1->r1 - set1->r0) * (set1->g1-set1->g0)
	       * (set1->b1 - set1->b0));
  set2->vol = ((set2->r1 - set2->r0) * (set2->g1 - set2->g0)
	       * (set2->b1 - set2->b0));
  return(1);
  }

/*
 *----------------------------------------------------------------------------
 *  CQ_Mark():
 *----------------------------------------------------------------------------
 */

static void CQ_Mark(struct cq_box *cube, int label, unsigned int *tag)
  {
  register int r, g, b;

  for(r = cube->r0 + 1; r <= cube->r1; r++)
    for(g = cube->g0 + 1; g <= cube->g1; g++)
      for(b = cube->b0 + 1; b <= cube->b1; b++)
        tag[(r << 10) + (r << 6) + r + (g << 5) + g + b] = label;
  }

