/*
 * fileSelectWidget
 *
 * Athena-based file browsing/selector widget (a descendant of Composite)
 *
 * v1.1 - Mark A. Lindner - April 1996
 *
 */

/* --- feature switches --- */

#define _POSIX_SOURCE 1

#define DEBUG

/* --- system headers --- */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/TextSrc.h>
#include <X11/Xaw/TextSink.h>
#include <X11/Xaw/AsciiSrc.h>
#include <X11/Xaw/AsciiSink.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Label.h>

#include <dirent.h>
#include <sys/types.h>
#include <sys/param.h>

/* --- local headers --- */

#include "geotouch.h"
#include "fselP.h"

/* --- macros --- */

#define INITIAL_WIDTH 300
#define INITIAL_HEIGHT 300
#define SPACING 10
#define MAX_PATHNAME 1024




/****************************************************************
 *
 * fsel Resources
 *
 ****************************************************************/

static XtResource resources[] =
  {
    {
    XtNcallback,
    XtCCallback,
    XtRCallback,
    sizeof(XtPointer),
    XtOffset(FileSelectWidget, fileSelect.callbacks),
    XtRCallback,
    (XtPointer)NULL
    },
    {
    XtNhSpace,
    XtCHSpace,
    XtRDimension,
    sizeof(Dimension),
    XtOffset(FileSelectWidget, fileSelect.h_space),
    XtRImmediate,
    (XtPointer)300
    },
    {
    XtNvSpace,
    XtCVSpace,
    XtRDimension,
    sizeof(Dimension),
    XtOffset(FileSelectWidget, fileSelect.v_space),
    XtRImmediate,
    (XtPointer)300
    },
    {
    XtNpathName,
    XtCPathName,
    XtRString,
    sizeof(String),
    XtOffset(FileSelectWidget, fileSelect.path),
    XtRString,
    (XtPointer)NULL
    }
  };

/****************************************************************
 *
 * Full class record constant
 *
 ****************************************************************/

static void fsel_initialize();
static void fsel_resize();
static void fsel_destroy();
static Boolean fsel_set_values();
static void fsel_change_managed();
static XtGeometryResult fsel_query_geometry(), fsel_geometry_manager();
/* ######################## */
/* ########## function definitions############## */

  static void fsel_readdir(Widget);
  static void fsel_select_file(Widget, XtPointer, XtPointer);
  static void fsel_cancel(Widget, XtPointer, XtPointer);
  static void fsel_chdir(Widget, XtPointer, XtPointer);
  static void fsel_check(char *);


FileSelectClassRec fileSelectClassRec =
  {
    {
    /* core_class fields */
    /* superclass         */    (WidgetClass) &compositeClassRec,
    /* class_name         */    "fileSelect",
    /* widget_size        */    sizeof(FileSelectRec),
    /* class_initialize   */    NULL,
    /* class_part_init    */	NULL,
    /* class_inited       */	FALSE,
    /* initialize         */    fsel_initialize,
    /* initialize_hook    */	NULL,
    /* realize            */    XtInheritRealize,
    /* actions            */    NULL,
    /* num_actions	  */	0,
    /* resources          */    resources,
    /* num_resources      */    XtNumber(resources),
    /* xrm_class          */    NULLQUARK,
    /* compress_motion	  */	TRUE,
    /* compress_exposure  */	TRUE,
    /* compress_enterleave*/	TRUE,
    /* visible_interest   */    FALSE,
    /* destroy            */    fsel_destroy,
    /* resize             */    fsel_resize,
    /* expose             */    NULL,
    /* set_values         */    fsel_set_values,
    /* set_values_hook    */	NULL,
    /* set_values_almost  */    XtInheritSetValuesAlmost,
    /* get_values_hook    */	NULL,
    /* accept_focus       */    NULL,
    /* version            */	XtVersion,
    /* callback_private   */    NULL,
    /* tm_table           */    NULL,
    /* query_geometry     */	fsel_query_geometry,
    /* display_accelerator*/	XtInheritDisplayAccelerator,
    /* extension          */	NULL
    },
    {
    /* composite_class fields */
    /* geometry_manager   */    fsel_geometry_manager,
    /* change_managed     */    fsel_change_managed,
    /* insert_child	  */	XtInheritInsertChild,
    /* delete_child	  */	XtInheritDeleteChild,
    /* extension          */	NULL
    },
    {
    /* fileSelect class fields */
    /* empty		  */	0
    }
  };

WidgetClass fileSelectWidgetClass = (WidgetClass)&fileSelectClassRec;

/****************************************************************
 *
 * Private Routines
 *
 ****************************************************************/

#define TEXT_ACCELERATORS \
  "#override\n<Key>Return: set() unset() notify()"

#define LIST_TRANSLATIONS \
  "<Btn1Down>(2): Set() Notify()\n<Btn1Down>,<Btn1Up>: Set()"



/*
 *  fsel_do_layout() : do the actual widget layout, or just calculate new
 *  positions (perhaps redundant)
 */

static void fsel_do_layout(Widget w, Boolean doit)
  {
  FileSelectWidget fsw = (FileSelectWidget)w;
  Widget w_port, w_text, w_cancel, w_chdir, w_label;
  Dimension portw, porth, textw, labelw; /* main window */
  Position portx, porty, textx, texty, chdirx, chdiry, cancelx, cancely,
    labelx, labely;

  w_port = fsw->fileSelect.w_port;
  w_text = fsw->fileSelect.w_text;
  w_chdir = fsw->fileSelect.w_chdir;
  w_cancel = fsw->fileSelect.w_cancel;
  w_label = fsw->fileSelect.w_label;

  /* size all 5 widgets so that space is fully utilized */

  portw = (fsw->core.width - (2 * SPACING));

  porth = (fsw->core.height - (4 * SPACING) - w_text->core.height -
	   (2 * w_text->core.border_width) - w_cancel->core.height -
	   (2 * w_cancel->core.border_width));

  textw = (fsw->core.width - (3 * SPACING) - w_chdir->core.width -
	   (2 * w_chdir->core.border_width));

  textx = texty = SPACING;

  chdirx = (fsw->core.width - SPACING - w_chdir->core.width -
	      (2 * w_chdir->core.border_width));

  chdiry = SPACING;

  portx = SPACING;
  porty = ((2 * SPACING) + (2 * w_text->core.border_width) +
	   w_text->core.height);

  cancelx = (fsw->core.width - SPACING - w_cancel->core.width -
	     (2 * w_cancel->core.border_width));
  cancely = (fsw->core.height - SPACING - w_cancel->core.height -
	     (2 * w_cancel->core.border_width));

  labelx = SPACING;
  labely = cancely;
  labelw = (fsw->core.width - (3 * SPACING) - w_cancel->core.width -
	    (2 * w_cancel->core.border_width));

  /* do the reorganization if we're supposed to */

  if(doit)
    {
    XtResizeWidget(w_port, portw, porth, 1);
    XtResizeWidget(w_text, textw, w_text->core.height, 1);
    XtResizeWidget(w_label, labelw, w_label->core.height, 0);

    XtMoveWidget(w_port, portx, porty);
    XtMoveWidget(w_text, textx, texty);
    XtMoveWidget(w_cancel, cancelx, cancely);
    XtMoveWidget(w_chdir, chdirx, chdiry);
    XtMoveWidget(w_label, labelx, labely);
    }
  }

/*
 *  fsel_geometry_manager(): geometry_manager method for widget
 */

static XtGeometryResult fsel_geometry_manager(Widget w,
  XtWidgetGeometry *request, XtWidgetGeometry *reply)
  {
  XtWidgetGeometry allowed;

  if(request->request_mode & ~(XtCWQueryOnly | CWWidth | CWHeight))
    return(XtGeometryNo);

  if(request->request_mode & CWWidth)
    allowed.width = request->width;
  else
    allowed.width = w->core.width;

  if(request->request_mode & CWHeight)
    allowed.height = request->height;
  else
    allowed.height = w->core.height;

  if(allowed.width == w->core.width && allowed.height == w->core.height)
    return(XtGeometryNo);

/*
  if(!(request->request_mode & XtCWQueryOnly))  // this is BAD.
    fsel_do_layout(w, False);
*/
  return(XtGeometryYes);
  }

/*
 *  fsel_query_geometry() : query_geometry method handler for widget
 */

static XtGeometryResult fsel_query_geometry(Widget w, 
  XtWidgetGeometry *request, XtWidgetGeometry *reply_return)
  {
  XtGeometryResult result = XtGeometryNo;

  request->request_mode &= CWWidth | CWHeight;

  if(!request->request_mode) return(XtGeometryYes);

  /* if proposed size is large enough, accept it; otherwise, suggest
     our arbitrary initial size. */

  if(request->request_mode & CWHeight)
    {
    if(request->height < INITIAL_HEIGHT)
      {
      result = XtGeometryAlmost;
      reply_return->height = INITIAL_HEIGHT;
      reply_return->request_mode &= CWHeight;
      }
    else result = XtGeometryYes;
    }

  if(request->request_mode & CWWidth)
    {
    if(request->width < INITIAL_WIDTH)
      {
      result = XtGeometryAlmost;
      reply_return->width = INITIAL_WIDTH;
      reply_return->request_mode &= CWWidth;
      }
    else result = XtGeometryYes;
    }
  return(result);
  }

/*
 *  fsel_resize() : resize method handler for widget
 */

static void fsel_resize(Widget w)
  {
  fsel_do_layout(w, True);
  }

/*
 *  fsel_change_managed() : change_managed method handler for widget
 */

static void fsel_change_managed(Widget w)
  {
  fsel_do_layout(w, True);
  }

/*
 *  fsel_initialize() : initialize method handler for widget
 */

static void fsel_initialize(Widget cur, Widget new)
  {
  FileSelectWidget newfsw = (FileSelectWidget)new;
  Widget w_port, w_list, w_text, w_chdir, w_cancel, w_label;
  WidgetList wl;
  Pixel fgcolor, bgcolor;


  if(!newfsw->core.width) newfsw->core.width = INITIAL_WIDTH;
  if(!newfsw->core.height) newfsw->core.height = INITIAL_HEIGHT;

  /* allocate a buffer to store the current pathname */

  newfsw->fileSelect.curpath = (char *)XtMalloc((MAXPATHLEN + 1)
						* sizeof(char));
  *(newfsw->fileSelect.curpath) = '\0';
  newfsw->fileSelect.editbuf = (char *)XtMalloc((MAXPATHLEN + 1)
						* sizeof(char));

  /* allocate a pointer array (initially 200 elements) for filenames */

  newfsw->fileSelect.listsize = 200;
  newfsw->fileSelect.listlen = 0;
  newfsw->fileSelect.sel = 0;
  newfsw->fileSelect.list = (char **)XtMalloc(200 * sizeof(char *));
  memset(newfsw->fileSelect.list, 0, 200 * sizeof(char *));

  /* This may be a problem; if you read the resource in and then try to MODIFY
     the buffer pointed to by the resource pointer, things will be corrupted.
     As long as you don't, it's fine: */

  strcpy(newfsw->fileSelect.curpath, "?");
  if(newfsw->fileSelect.path)
    {
    strncpy(newfsw->fileSelect.curpath, newfsw->fileSelect.path, MAXPATHLEN);
    *(newfsw->fileSelect.curpath + MAXPATHLEN - 1) = '\0';
    }
  fsel_check(newfsw->fileSelect.curpath);  

  newfsw->fileSelect.path = newfsw->fileSelect.curpath;

  strcpy(newfsw->fileSelect.editbuf, newfsw->fileSelect.curpath);

  /* create all the widgets */

  XtVaGetValues((Widget)newfsw, XtNforeground, &fgcolor,
		XtNbackground, &bgcolor, NULL);

  w_port = XtVaCreateManagedWidget("w_port", viewportWidgetClass,
				   (Widget)newfsw,
				   XtNallowVert, True, XtNforceBars, True,
				   XtNuseRight, True,
				   XtNforeground, fgcolor,
				   XtNbackground, bgcolor,
				   NULL);
  newfsw->fileSelect.w_port = w_port;

  XtVaGetValues(w_port, XtNchildren, &wl, NULL);
  newfsw->fileSelect.w_vbar = wl[1];

  w_list = XtVaCreateManagedWidget("w_list", listWidgetClass, w_port,
				   XtNdefaultColumns, 1, XtNverticalList, True,
				   XtNlist, newfsw->fileSelect.list,
				   XtNtranslations,
				   XtParseTranslationTable(LIST_TRANSLATIONS),
				   XtNforeground, fgcolor,
				   XtNbackground, bgcolor,
				   NULL);
  newfsw->fileSelect.w_list = w_list;

  XtAddCallback(w_list, XtNcallback, fsel_select_file, (XtPointer)newfsw);

  w_text = XtVaCreateManagedWidget("w_text", asciiTextWidgetClass,
				   (Widget)newfsw,
				   XtNeditType, XawtextEdit,
				   XtNlength, MAX_PATHNAME,
				   XtNuseStringInPlace, True,
				   XtNstring, newfsw->fileSelect.editbuf,
				   XtNforeground, fgcolor,
				   XtNbackground, bgcolor,
				   NULL);
  newfsw->fileSelect.w_text = w_text;

  w_cancel = XtVaCreateManagedWidget("w_cancel", commandWidgetClass,
				     (Widget)newfsw,
				     XtNlabel, "Cancel",
				     XtNcornerRoundPercent, 35,
				     XtNshapeStyle, XmuShapeRoundedRectangle,
				     XtNforeground, fgcolor,
				     XtNbackground, bgcolor,
				     NULL);
  newfsw->fileSelect.w_cancel = w_cancel;

  XtAddCallback(w_cancel, XtNcallback, fsel_cancel, (XtPointer)newfsw);

  w_chdir = XtVaCreateManagedWidget("w_chdir", commandWidgetClass,
				    (Widget)newfsw,
				    XtNlabel, "New Path", XtNborderWidth, 2,
				    XtNcornerRoundPercent, 35,
				    XtNshapeStyle, XmuShapeRoundedRectangle,
				    XtNaccelerators,
				    XtParseAcceleratorTable(TEXT_ACCELERATORS),
				    XtNforeground, fgcolor,
				    XtNbackground, bgcolor,
				    NULL);
  newfsw->fileSelect.w_chdir = w_chdir;


  XtInstallAccelerators(w_chdir, w_text);



  XtAddCallback(w_chdir, XtNcallback, fsel_chdir, (XtPointer)newfsw);

  newfsw->fileSelect.w_text = w_text;

  w_label = XtVaCreateManagedWidget("w_label", labelWidgetClass,
				    (Widget)newfsw,
				    XtNborderWidth, 0,
				    XtNjustify, XtJustifyLeft,
				    XtNforeground, fgcolor,
				    XtNbackground, bgcolor,
				    NULL);

  newfsw->fileSelect.w_label = w_label;

  XtInstallAccelerators(w_text, w_chdir);

  /* read the initial directory */

  fsel_readdir((Widget)newfsw);
  }

/*
 *  fsel_set_values() : set_values method handler for widget
 */

static Boolean fsel_set_values(Widget current, Widget request, Widget new,
			       ArgList args, Cardinal *num_args)
  {
  FileSelectWidget fswcurrent = (FileSelectWidget)current;
  FileSelectWidget fswnew = (FileSelectWidget)new;
  Boolean r = False;


  /* check pathname...is it too long, or invalid? */

  strcpy(fswnew->fileSelect.curpath, fswnew->fileSelect.path), r = True;
  fsel_check(fswnew->fileSelect.curpath);

  /* need to redo layout if h_space or v_space change */

  if((fswnew->fileSelect.h_space != fswcurrent->fileSelect.h_space) ||
      (fswnew->fileSelect.v_space != fswcurrent->fileSelect.v_space))
    fsel_do_layout((Widget)fswnew, True);

  XtVaSetValues(fswnew->fileSelect.w_text, XtNstring,
		fswnew->fileSelect.curpath, NULL);

  fsel_readdir(new);

  return(r);
  }
/*
 *  fsel_compare() : comparison function for sorting filenames
 */
 


int fsel_compare(char **a, char **b)
  {
  return(strcmp(*a, *b));
  }



/*
 *  fsel_readdir() : read the current directory and update widgets
 */

static void fsel_readdir(Widget w)
  {
  FileSelectWidget fsw = (FileSelectWidget)w;
  struct dirent *de;
  struct stat buf;
  DIR *dp;
  int i, l, nfiles = 0, ndirs = 0;
  char *s, label[30];

 
  /* free all allocated filename buffers */

  for(i = 0; i < fsw->fileSelect.listlen; i++)
    if(fsw->fileSelect.list[i])
      {
      XtFree(fsw->fileSelect.list[i]);
      fsw->fileSelect.list[i] = NULL;
      }

  /* downsize the buffer to 200 pointers if it's bigger than that */

  if(fsw->fileSelect.listsize > 200)
    {
    fsw->fileSelect.list = (char **)XtRealloc((char *)fsw->fileSelect.list,
					      200);
    fsw->fileSelect.listsize = 200;
    }

  /* read the current directory, skipping dotfiles and special files */

  chdir(fsw->fileSelect.curpath); /* this shouldn't ever fail */

  i = 0;
  if(!(dp = opendir(".")))
    {
    puts("\a");
#ifdef DEBUG
    fprintf(stderr, "fileSelect: Permission error trying to read directory\n");
#endif
    return;
    }
  while((de = readdir(dp)))
    {
    if(!de->d_ino) continue;
    if(lstat(de->d_name, &buf))
      {
#ifdef DEBUG
      fprintf(stderr, "fileSelect: stat failed on file: %s\n", de->d_name);
#endif
      continue;
      }
    if(!(S_ISDIR(buf.st_mode) || S_ISREG(buf.st_mode))) continue;
    if((*(de->d_name) == '.') &&
       (strcmp(de->d_name, ".."))) continue; /* skip dot files */
    l = strlen(de->d_name);
    s = (char *)XtMalloc((l+2) * sizeof(char));
    strcpy(s, de->d_name);
    fsw->fileSelect.list[i] = s;

    /* append a '/' if it's a dir; update counters */

    if(S_ISDIR(buf.st_mode))
      {
      ndirs++;
      *(s+l) = '/';
      }
    else
      {
      nfiles++;
      *(s+l) = '\0';
      }
    *(s+l+1) = '\0';

    /* take care of reallocation of filename pointer array */

    if(!(++i % 200))
      {
      fsw->fileSelect.listsize += 200;
      fsw->fileSelect.list = (char **)XtRealloc((char *)fsw->fileSelect.list,
						fsw->fileSelect.listsize *
						sizeof(char *));
      }
    }
  closedir(dp);
  fsw->fileSelect.listlen = i;
  fsw->fileSelect.list[i] = NULL;

  /* now sort the list alphabetically with qsort */

  qsort(fsw->fileSelect.list, fsw->fileSelect.listlen, sizeof(char *),
	fsel_compare);

  /* update all affected widget resources */

  XawScrollbarSetThumb(fsw->fileSelect.w_vbar, 0, -1);

  XawListChange(fsw->fileSelect.w_list, fsw->fileSelect.list,
		fsw->fileSelect.listlen, MAX_PATHNAME, True);
  XawListHighlight(fsw->fileSelect.w_list, 0);
  fsw->fileSelect.sel = 0;

  ndirs--;
  sprintf(label, "%i file%s, %i folder%s",
	  nfiles, (nfiles != 1 ? "s" : ""),
	  ndirs, (ndirs != 1 ? "s" : ""));
  XtVaSetValues(fsw->fileSelect.w_label, XtNlabel, label, NULL);
  }

/*
 *  fsel_select_file() : callback function to handle file selection
 */

static void fsel_select_file(Widget w, XtPointer clidat,
			     XtPointer calldat)
  {
  FileSelectWidget fsw = (FileSelectWidget)clidat;
  XawListReturnStruct *sel = (XawListReturnStruct *)calldat;
  int l, ll;
  char *p;

  void fsel_check(char *);
  Boolean fsel_check_dir(char *);

  /* see if current path is a file rather than directory path;
   * thus if this widget has residue from a previous invocation, then it will
   * just chop the filename off the end of the path and then check the
   * resulting path for validity
   */

  ll = strlen(fsw->fileSelect.curpath);
  if(*(fsw->fileSelect.curpath + --ll) != '/')
    {
    if((p = strrchr(fsw->fileSelect.curpath, '/')))
      {
      *p = '\0';
      fsel_check_dir(fsw->fileSelect.curpath);
      }
    }

  /* find out what was selected, and whether it was a directory */

  fsw->fileSelect.sel = sel->list_index;
  l = strlen(sel->string);

  /* if they chose a directory and it's not the root dir... */

  if((*(sel->string + l - 1)) == '/' && (l > 1))
    {
    /* if they selected "..", back up one directory level */

    if(!strcmp(sel->string, "../"))
      {
      if(strcmp(fsw->fileSelect.curpath, "/"))
	{
	if((p = strrchr(fsw->fileSelect.curpath, '/'))) *p = '\0';
	if((p = strrchr(fsw->fileSelect.curpath, '/'))) *(++p) = '\0';
	}
      }
    else
      {
      /* otherwise if new path wouldn't be too long, append new directory */

      if((strlen(fsw->fileSelect.curpath) + strlen(sel->string)) >=
	 MAXPATHLEN)
	{
	puts("\a");
#ifdef DEBUG
        fprintf(stderr, "fileSelect: path is getting too long (max 1024)\n");
#endif
	}
      else
	{
	/* if we can access the directory, switch; otherwise don't */

	if(!fsel_check_dir(sel->string))
	  {
	  puts("\a");
#ifdef DEBUG
	  fprintf(stderr, "fileSelect: access denied on directory\n");
#endif
	  }
	else strcat(fsw->fileSelect.curpath, sel->string);
	}
      }

    /* set the resource and call ourselves again to read the new directory */

    XtVaSetValues(fsw->composite.children[1], XtNstring,
		  fsw->fileSelect.curpath, NULL);
    
    *(sel->string) = 0;
    fsel_readdir((Widget)fsw);
    }

  /* else if they chose a plain file... */

  else
    {
    /* complain if the path is too long now */

    if((strlen(fsw->fileSelect.curpath) + strlen(sel->string)) >=
       MAX_PATHNAME)
      {
      puts("\a");
#ifdef DEBUG
      fprintf(stderr, "fileSelect: path is getting too long (max 1024)\n");
#endif
      }
    else
      {
      /* otherwise append the file to the path and call our callbacks */

      strcat(fsw->fileSelect.curpath, sel->string);

      fsw->fileSelect.path = fsw->fileSelect.curpath;

      XtPopdown(XtParent((Widget)fsw));
      XtCallCallbacks((Widget)fsw, XtNcallback, (XtPointer)True);
      }
    }
  }

/*
 *  fsel_chdir() : callback function to handle accepting the path
 */

static void fsel_chdir(Widget w, XtPointer clidat, XtPointer calldat)
  {
  FileSelectWidget fsw = (FileSelectWidget)clidat;
  char *p;


  /* get the path */

  XtVaGetValues(fsw->fileSelect.w_text, XtNstring, &p, NULL);

  /* if we can access the directory, switch; otherwise don't */

  fsel_check(p);

  XtVaSetValues(fsw->fileSelect.w_text, XtNstring, p, NULL);

  fsel_readdir((Widget)fsw);
  }

/*
 *  fsel_cancel() : callback function to handle cancel button
 */

static void fsel_cancel(Widget w, XtPointer clidat, XtPointer calldat)
  {
  FileSelectWidget fsw = (FileSelectWidget)clidat;

  /* call our callbacks with -1 to let them know there was a cancel */

  XtPopdown(XtParent((Widget)fsw));
  XtCallCallbacks((Widget)fsw, XtNcallback, (XtPointer)False);
  }

/*
 *  fsel_destroy() : destroy method handler for widget
 */

static void fsel_destroy(Widget w)
  {
  FileSelectWidget fsw = (FileSelectWidget)w;
  int i;

  /* free the path & edit buffers */

  XtFree(fsw->fileSelect.curpath);
  XtFree(fsw->fileSelect.editbuf);

  /* free all filenames in the pointer array */

  for(i = 0; i < (fsw->fileSelect.listlen); i++)
    XtFree(fsw->fileSelect.list[i]);

  /* free the pointer array itself */

  XtFree((char *)fsw->fileSelect.list);
  }


/*
 *  makes sure s is an ok dir and is accessible; if it isn't, it tries again
 *  on the current working directory. If *that* fails, it goes back to root
 *  directory.
 */

void fsel_check(char *s)
  {
  Boolean fsel_check_dir(char *);

  if(fsel_check_dir(s)) return;
  else
    {
    if(getcwd(s, MAXPATHLEN))
      {
      *(s + MAXPATHLEN - 1) = '\0';
      if(fsel_check_dir(s)) return;
      }
    }
  strcpy(s, "/");
  }

Boolean fsel_check_dir(char *s)
  {
  char *p;
  size_t l;
  struct stat buf;

  /* is it too long? */

  if((l = strlen(s)) >= MAXPATHLEN)
    return(False);

  /* is it missing final slash? */

  if(*(p = (s + l - 1)) != '/')
    {
    *(++p) = '/';
    *(++p) = '\0';
    }

  /* can we get to it? */

  if(access(s, R_OK|X_OK)) return(False);

  /* is it a directory? */

  if(lstat(s, &buf)) return(False);
  if(!S_ISDIR(buf.st_mode)) return(False);

  return(True);
  }
