
/* REFERENCE:
some of these projections are taken from
"Map Projections - A Working Manual"
USGS Prof Paper 1395
John P. Snyder

*/

/* --- feature switches --- */

#define _POSIX_SOURCE 1

/* --- system headers --- */

/* --- local headers --- */

#include "geotouch.h"
#include "primitive.h"

/* --- macros --- */

#define A	6378206.4
#define E2	0.00676866
#define E	0.0822719
#define E1	0.993231340
#define TwoE	0.164543800
#define R	6378.2064
#define DEG2RAD	0.017453293
#define RAD2DEG	57.295778667

/* --- global variables --- */

static double EN, C, B, RHO0, PHI0, LAM0, PHI1;
static double   CPHI1, SPHI1;

/* --- functions --- */

/*****  ALBERS EA   SPHERICAL    **************/

void set_albers_EA_sphere(float phi1, float phi2, float phi0, float lam0)
  {

  EN = (sin(DEG2RAD * phi2) + sin(DEG2RAD * phi1)) / 2.0;
  C = cos(DEG2RAD * phi1) * cos(DEG2RAD * phi1) + EN * 2 * sin(DEG2RAD * phi1);
  RHO0 = R * sqrt(C - 2 * EN * sin(DEG2RAD * phi0)) / EN;
  PHI0 = phi0;
  LAM0 = lam0;

  /* fprintf(stderr,"in set: EN=%f C=%f RHO0=%f \n", EN, C, RHO0); */
  }

void albers_EAsphr_xy(float phi, float lam, float *x, float *y)
  {
  double rho, theta;

  rho = R * sqrt(C - 2 * EN * sin(DEG2RAD * phi)) / EN;
  theta = EN * (DEG2RAD * lam - DEG2RAD * LAM0);
  *x = rho * sin(theta);
  *y = RHO0 - rho * cos(theta);
  }

void albers_EAsphr_ll(float *phi, float *lam, float x, float y)
  {
  double rho, theta;

  rho = sqrt((sqr(x) + (RHO0 - y) * (RHO0 - y)));

  if(EN < 0)
    {
    /* see top of page 101 in snyder's book on map projections  */

    theta = RAD2DEG * atan2(-x, (-RHO0) - (-y));
    }
  else theta = RAD2DEG * atan2(x, RHO0 - y);

  /*
   * fprintf(stderr,"x=%f y=%f rho=%f theta=%f C=%f  EN=%f R=%f LAM0=%f
   * RHO0=%f\n",x,y, rho, theta, C, EN, R, LAM0, RHO0);
   */

  *phi = RAD2DEG * asin((C - (rho * EN / R) * (rho * EN / R)) / (2 * EN));
  *lam = (LAM0 + theta / EN);
  }

/******  MERCATOR   SPHERICAL    **************/

void set_merc_sphere(float lam0)
  {
  LAM0 = lam0;
  
  }

void merc_sphr_xy(float phi, float lam, float *x, float *y)
  {
  double theta = (lam - LAM0);

  if (theta < 180) theta += 360;
  if (theta > 180) theta -= 360;

  *x = DEG2RAD * R * theta;
  *y = R * log(tan(DEG2RAD * (45 + phi / 2)));
  }

void merc_sphr_ll(float *phi, float *lam, float x, float y)
  {
  *phi = 90 - RAD2DEG * 2 * atan(exp(-y / R));
  *lam = RAD2DEG * (x / R) + LAM0;
  }


/******  LAMBERT   SPHERICAL    **************/
/*  Lambert Azimuthal Equal -Area Projection  */
/****   Snyder P. 185  ***/
void set_Lambert_Az_sphere( float phi1, float lam0)
  {

    fprintf(stderr,"set_Lambert_Az_sphere :   lam0=%f   phi1=%f\n", lam0, phi1);
    
  LAM0 = lam0;
  PHI1 = DEG2RAD *phi1;
  CPHI1 = cos(PHI1);
  SPHI1 = sin(PHI1);
  }

void Lambert_Az_sphr_xy(float phi, float lam, float *x, float *y)
  {
  double kp, theta = (lam - LAM0);
  double phirad;
  
  if (theta < 180) theta += 360;
  if (theta > 180) theta -= 360;

  theta = DEG2RAD*theta;
  phirad = DEG2RAD*phi;

  kp = sqrt(2/(1+SPHI1*sin(phirad)+CPHI1*cos(phirad)*cos(theta)));
 
  

  *x = R * kp * cos(phirad)*sin(theta);
  *y = R * kp * (CPHI1*sin(phirad) - SPHI1*cos(phirad)*cos(theta));
  }

void Lambert_Az_sphr_ll(float *phi, float *lam, float x, float y)
  {
    double rho , c;
    
    rho = sqrt(x*x+y*y);
    if(rho == 0.0)
      {
	*phi =  RAD2DEG*PHI1;
	*lam = LAM0;
	return;
      }
    c = 2*atan(rho/(2*R));
    
    *phi = RAD2DEG*(asin(cos(c)*SPHI1+(y*sin(c)*CPHI1/rho)));
    *lam = LAM0 + RAD2DEG*(atan(x*sin(c)/(rho*CPHI1*cos(c) - y*SPHI1*sin(c))));
  }

/******  MILLER   SPHERICAL    ***snyder p. 88***********/

void set_miller_sphere(float lam0)
  {
  LAM0 = lam0;
  }

void miller_sphr_xy(float phi, float lam, float *x, float *y)
  {
  double theta = (lam - LAM0);

  if(theta < 180) theta += 360;
  if (theta > 180) theta -= 360;
  *x = DEG2RAD * R * theta;
  *y = ((R / 1.6) * log((1 + sin(0.8 * DEG2RAD * phi))
			/ (1 - sin(0.8 * DEG2RAD * phi))));
  }

void miller_sphr_ll(float *phi, float *lam, float x, float y)
  {

  *phi = RAD2DEG * (2.5 * atan(exp(0.8 * y / R)) - 1.963495408);
  *lam = RAD2DEG *(x / R) + LAM0;
  }

/************1 0 CROSSON PROJOECTIONs  *************/

static float FR(float t)
  {
  float r = sin(t);

  return (1.0 - 0.0033670033 * sqr(r));
  }

void set_ngclc(float phi, float lam)
  {
  double a, rho, b, c, cdist, crlat, zgcl;

  PHI0 = DEG2RAD * phi;
  LAM0 = DEG2RAD * lam;

  crlat = atan(ECCEN * sin(2.0 * PHI0) / FR(PHI0));
  zgcl = PHI0 - crlat;
  a = FR(zgcl);
  rho = EARTHRAD * a;
  b = sqr((ECCEN * sin(2.0 * zgcl) / a)) + 1.0;
  c = 2.0 * ECCEN * cos(2.0 * zgcl) * a + sqr((ECCEN * sin(2.0 * zgcl)));
  cdist = c / (sqr(a) * b) + 1.0;
  RHO0 = rho;
  C = crlat;
  B = cdist;
  }

void ngclc(float phi, float lam, float *x, float *y)
  {
  double xlat, ylon, ysign = -1.0;

  xlat = DEG2RAD * phi;
  ylon = DEG2RAD * lam;
  *y = ysign*RHO0 * (LAM0 - ylon) * cos(xlat - C);
  *x = RHO0 * (xlat - PHI0) / B;
  }

void nlcgc(float *phi, float *lam, float ex, float why)
  {
  double xlat,  ysign = -1.0;

  xlat = (ex) * B / RHO0 + PHI0;
  /* ylon = LAM0 - (why) / (RHO0 * cos(xlat - C));*/

  *phi = RAD2DEG * (ex * B / RHO0 + PHI0);
  *lam = RAD2DEG * (LAM0 - (why*ysign) / (RHO0 * cos(xlat - C)));
  }

/*****************************************************/

void set_no_proj(float phi, float lam)
  {

  PHI0 = phi;
  LAM0 = lam;

  fprintf(stderr,"setting NO projection: phi0=%f lam0=%f\n", PHI0,LAM0);
  }

void no_proj_2lc(float phi, float lam, float *x, float *y)
  {

  *x =  lam - LAM0;
  *y =  phi - PHI0;
  }

void no_proj_2gc(float *phi, float *lam, float ex, float why)
  {

  *lam = ex + LAM0;
  *phi = why + PHI0;
  }

/*****************************************************/
