/*
 * ---------------------------------------------------------------------------
 * rotate.c - Rotation shell source
 *
 * Mark Lindner - 2/10/96
 * ---------------------------------------------------------------------------
 */

/* --- feature test switches --- */

#define _POSIX_SOURCE 1

/* --- system headers --- */

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/Repeater.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/TextSrc.h>
#include <X11/Xaw/TextSink.h>
#include <X11/Xaw/AsciiSrc.h>
#include <X11/Xaw/AsciiSink.h>

#include <stdio.h>
#include <math.h>

/* --- local headers --- */

#include "rotate.h"

/* --- macros --- */

#define ROT_HELPWIDTH 600
#define ROT_HELPHEIGHT 250

#define ROT_DFL_WIDTH 500
#define ROT_DFL_HEIGHT 500

#define ROT_DFL_INCREMENT 2.0

/* default parameters for dialog boxes */

#define ROT_MAX_PARAM 30
#define ROT_DFL_X0 0.0
#define ROT_DFL_Y0 0.0
#define ROT_DFL_Z0 0.0
#define ROT_DFL_XR 0.0
#define ROT_DFL_YR 0.0
#define ROT_DFL_ZR 0.0

/* --- global variables --- */

static float rot_dfl_radius = 0;

static const char *rot_helpstring = "\
[Output...]      Output the dataset to the screen or a file.\n\
[Fullscreen]     Toggle full-screen mode.\n\
[x<] and [x>]    Rotate image about x-axis (horizontally wrt. screen)\n\
[y<] and [y>]    Rotate image about y-axis (vertically wrt. screen)\n\
[z<] and [z>]    Rotate image about z-axis (into screen)\n\
[Zoom]           Adjust radius and origin of display sphere\n\
[Rotate]         Adjust rotation angles about the x, y, and z axes\n\
[Reset]          Reset all parameters to initial values\n\
[Fancy]          Display points color-coded by distance from observer\n\
[Show Angles]    Toggle display of angles between absolute z-axis and current axes\n\
[Show Container] Toggle display of pointset container box\n\
[Help]           Get help dialog\n\
[Dismiss]        Dismiss the Spin Events dialog";

/* --- global functions --- */

static void rot_set_rotate_resource(Widget, float, float, float);
static void rot_set_zoom_resource(Widget, float, float, float, float);

/* --- functions --- */
  extern void rot_output_ps(Widget, XtPointer, XtPointer);
  static void rot_output_screen(Widget, XtPointer, XtPointer);
  static void rot_fullscreen(Widget, XtPointer, XtPointer);
  static void rot_zoom(Widget, XtPointer, XtPointer);
  static void rot_rotate(Widget, XtPointer, XtPointer);
  static void rot_reset(Widget, XtPointer, XtPointer);
  static void rot_help(Widget, XtPointer, XtPointer);
 /*  static void rot_exit(Widget, XtPointer, XtPointer); */

  static void rot_rotxl(Widget, XtPointer, XtPointer);
  static void rot_rotxr(Widget, XtPointer, XtPointer);
  static void rot_rotyl(Widget, XtPointer, XtPointer);
  static void rot_rotyr(Widget, XtPointer, XtPointer);
  static void rot_rotzl(Widget, XtPointer, XtPointer);
  static void rot_rotzr(Widget, XtPointer, XtPointer);

  static void rot_zoom_remove(Widget, XtPointer, XtPointer);
  static void rot_zoom_apply(Widget, XtPointer, XtPointer);
  static void rot_helpremove(Widget, XtPointer, XtPointer);
  static void rot_rotate_remove(Widget, XtPointer, XtPointer);
  static void rot_rotate_apply(Widget, XtPointer, XtPointer);

  static void rot_fancy(Widget, XtPointer, XtPointer);
  static void rot_show_angles(Widget, XtPointer, XtPointer);
  static void rot_container(Widget, XtPointer, XtPointer);
/*
 * rot_main() gets called with pointset and lineset data and a pointer
 * to the widget that is its parent.
 */

Widget m2_container, m2_angles;

Widget rot_main(ptset_t *pointset, ptset_t *lineset, Widget top)
  {
  Widget *yoke = (Widget *)XtMalloc(sizeof (Widget) * 12);
  Widget plot, form, box, b_output, b_rotxl, b_rotxr, b_rotyl,
    b_rotyr, b_rotzl, b_rotzr, b_zoom, b_rotate, b_help, b_exit, b_reset,
    b_options, viewport, menu, m1_screen, m1_postscript,
    menu2, b_fullscreen, m2_fancy;
  Cursor cursor;
  Widget topshell, helpshell, helpbox, helptextbox, helpdone;
  Widget zoomshell, zoombox, zoomform, zoomlabel, zoomdone, zoomapply,
    zoomrlabel, zoomolabelx, zoomolabely, zoomolabelz, zoomsetox,
    zoomsetoy, zoomsetoz, zoomsetr;
  Widget rotateshell, rotdone, rotform, rotlabel, rotbox, rotapply, rotxlabel,
    rotylabel, rotzlabel, rotsetx, rotsety, rotsetz;
  char text[80];
  int bc;
  float minx, maxx, miny, maxy;




  /* set up the top-level shells */


  /* fprintf(stderr,"start rot_main\n"); */

  topshell = XtVaCreatePopupShell("topShell", transientShellWidgetClass,
    top, XtNx, ((WidthOfScreen(XtScreen(top)) - ROT_DFL_WIDTH) / 2),
    XtNy, ((HeightOfScreen(XtScreen(top)) - ROT_DFL_HEIGHT) / 2),
    XtNwidth, ROT_DFL_WIDTH, XtNheight, ROT_DFL_HEIGHT,
    XtNtitle, "Spin Events", NULL);

  yoke[10] = topshell;

  helpshell = XtVaCreatePopupShell("helpDialog", transientShellWidgetClass,
    topshell, XtNx, ((WidthOfScreen(XtScreen(top)) - ROT_HELPWIDTH) / 2),
    XtNy, ((HeightOfScreen(XtScreen(top)) - ROT_HELPHEIGHT) / 2),
    XtNwidth, ROT_HELPWIDTH, XtNheight, ROT_HELPHEIGHT,		     
    XtNtitle, "Spin Events Help", NULL);

  zoomshell = XtVaCreatePopupShell("zoomShell", transientShellWidgetClass,
    topshell, XtNx, 300, XtNy, 300, XtNtitle, "Set Zoom Parameters",
    XtNallowShellResize, False, NULL);

  rotateshell = XtVaCreatePopupShell("rotateShell", transientShellWidgetClass,
    topshell, XtNx, 300, XtNy, 300, XtNtitle, "Set Rotation Parameters",
    XtNallowShellResize, False, NULL);

  /* set up the top level widgets */

  form = XtVaCreateManagedWidget("form", formWidgetClass, topshell,
    XtNborderWidth, 0, XtNorientation, XtorientVertical, NULL);

  box = XtVaCreateManagedWidget("box", boxWidgetClass, form,
    XtNborderWidth, 2, XtNhSpace, 5, XtNbottom, XtChainTop,
    XtNright, XtChainRight, XtNorientation, XtorientHorizontal,
    XtNwidth, ROT_DFL_WIDTH, XtNheight, 60, NULL);

  cursor = XCreateFontCursor(XtDisplay(top), XC_right_ptr);

  /* make Output... menu */
/* fprintf(stderr,"make Output... menu\n"); */


  b_output = XtVaCreateManagedWidget("b_output", menuButtonWidgetClass, box,
				     XtNlabel, "Output...", NULL);
  XtVaGetValues(b_output, XtNbackground, &bc, NULL);

  menu = XtVaCreatePopupShell("menu", simpleMenuWidgetClass, b_output,
    XtNrightMargin, 15, XtNleftMargin, 15, XtNcursor, cursor,
    XtNbackground, bc, NULL);

  m1_screen = XtVaCreateManagedWidget("m1_screen", smeBSBObjectClass, menu,
    XtNlabel, "Coordinates", NULL);

  m1_postscript = XtVaCreateManagedWidget("m1_postscript", smeBSBObjectClass,
    menu, XtNlabel, "Postscript", NULL);

  /* make Options... menu */
/* fprintf(stderr,"make Options... menu\n"); */

  b_options = XtVaCreateManagedWidget("b_options", menuButtonWidgetClass, box,
				     XtNlabel, "Options...", NULL);

  menu2 = XtVaCreatePopupShell("menu", simpleMenuWidgetClass, b_options,
    XtNrightMargin, 15, XtNleftMargin, 15, XtNcursor, cursor,
    XtNbackground, bc, NULL);

  m2_fancy = XtVaCreateManagedWidget("m2_fancy", smeBSBObjectClass, menu2,
				   XtNlabel, "Fancy Plot", NULL); 

  m2_angles = XtVaCreateManagedWidget("m2_angles", smeBSBObjectClass, menu2,
				     XtNlabel, "Show Angles", NULL);

  m2_container = XtVaCreateManagedWidget("m2_container", smeBSBObjectClass,
    menu2, XtNlabel, "Show Container", NULL);

  /* make other buttons */
/* fprintf(stderr,"make other buttons  \n"); */

  b_fullscreen = XtVaCreateManagedWidget("b_fullscreen", toggleWidgetClass,
    box, XtNlabel, "Fullscreen", NULL);

  b_rotxl = XtVaCreateManagedWidget("b_rotxl", repeaterWidgetClass, box,
				    XtNlabel, "x<",
				    XtNshapeStyle, XmuShapeRoundedRectangle,
				    XtNinitialDelay, 400, XtNrepeatDelay,
				    50, XtNdecay, 0, NULL);

  b_rotxr = XtVaCreateManagedWidget("b_rotxr", repeaterWidgetClass, box,
				    XtNlabel, "x>", XtNflash, True,
				    XtNshapeStyle, XmuShapeRoundedRectangle,
				    XtNinitialDelay, 400, XtNrepeatDelay,
				    50, XtNdecay, 0, NULL);

  b_rotyl = XtVaCreateManagedWidget("b_rotyl", repeaterWidgetClass, box,
				    XtNlabel, "y<", XtNflash, True,
				    XtNshapeStyle, XmuShapeRoundedRectangle,
				    XtNinitialDelay, 400, XtNrepeatDelay,
				    50, XtNdecay, 0, NULL);

  b_rotyr = XtVaCreateManagedWidget("b_rotyr", repeaterWidgetClass, box,
				    XtNlabel, "y>", XtNflash, True,
				    XtNshapeStyle, XmuShapeRoundedRectangle,
				    XtNinitialDelay, 400, XtNrepeatDelay,
				    50, XtNdecay, 0, NULL);

  b_rotzl = XtVaCreateManagedWidget("b_rotzl", repeaterWidgetClass, box,
				    XtNlabel, "z<", XtNflash, True,
				    XtNshapeStyle, XmuShapeRoundedRectangle,
				    XtNinitialDelay, 400, XtNrepeatDelay,
				    50, XtNdecay, 0, NULL);

  b_rotzr = XtVaCreateManagedWidget("b_rotzr", repeaterWidgetClass, box,
				    XtNlabel, "z>", XtNflash, True,
				    XtNshapeStyle, XmuShapeRoundedRectangle,
				    XtNinitialDelay, 400, XtNrepeatDelay,
				    50, XtNdecay, 0, NULL);

  b_zoom = XtVaCreateManagedWidget("b_zoom", commandWidgetClass, box,
				   XtNlabel, "Zoom", NULL);

  b_rotate = XtVaCreateManagedWidget("b_rotate", commandWidgetClass, box,
				   XtNlabel, "Rotate", NULL);

  b_reset = XtVaCreateManagedWidget("b_reset", commandWidgetClass, box,
				   XtNlabel, "Reset", NULL);

  b_help = XtVaCreateManagedWidget("b_help", commandWidgetClass, box,
				   XtNlabel, "Help", NULL);

  b_exit = XtVaCreateManagedWidget("b_exit", commandWidgetClass, box,
				   XtNlabel, "Dismiss", NULL);

  cursor = XCreateFontCursor(XtDisplay(top), XC_cross);

  viewport = XtVaCreateManagedWidget("viewport", viewportWidgetClass, form,
    XtNwidth, ROT_DFL_WIDTH, XtNheight, ROT_DFL_HEIGHT,
    XtNfromVert, box, XtNcursor, cursor, XtNborderWidth, 2, NULL);

  plot = XtVaCreateManagedWidget("plot", rotateWidgetClass, viewport,
    XtNpointSet, pointset, XtNlineSet, lineset, NULL);

  XtVaGetValues(plot, XtNradius, &rot_dfl_radius, NULL);

  /* set the callbacks */

  XtAddCallback(b_zoom, XtNcallback, (XtCallbackProc)rot_zoom, zoomshell);
  XtAddCallback(b_rotate, XtNcallback, (XtCallbackProc)rot_rotate,
		rotateshell);

  XtAddCallback(b_reset, XtNcallback, (XtCallbackProc)rot_reset, yoke);

  XtAddCallback(b_help, XtNcallback, (XtCallbackProc)rot_help, helpshell);
  XtAddCallback(b_exit, XtNcallback, (XtCallbackProc)rot_exit, yoke);

  XtAddCallback(b_rotxl, XtNcallback, (XtCallbackProc)rot_rotxl, yoke);
  XtAddCallback(b_rotxr, XtNcallback, (XtCallbackProc)rot_rotxr, yoke);
  XtAddCallback(b_rotyl, XtNcallback, (XtCallbackProc)rot_rotyl, yoke);
  XtAddCallback(b_rotyr, XtNcallback, (XtCallbackProc)rot_rotyr, yoke);
  XtAddCallback(b_rotzl, XtNcallback, (XtCallbackProc)rot_rotzl, yoke);
  XtAddCallback(b_rotzr, XtNcallback, (XtCallbackProc)rot_rotzr, yoke);

  XtAddCallback(m1_screen, XtNcallback, (XtCallbackProc)rot_output_screen,
		yoke);
  XtAddCallback(m1_postscript, XtNcallback, (XtCallbackProc)rot_output_ps,
		yoke);


  XtAddCallback(m2_angles, XtNcallback, (XtCallbackProc)rot_show_angles, yoke);
  XtAddCallback(m2_container, XtNcallback, (XtCallbackProc)rot_container,
		yoke);
  XtAddCallback(m2_fancy, XtNcallback, (XtCallbackProc)rot_fancy, yoke);
  XtAddCallback(b_fullscreen, XtNcallback, (XtCallbackProc)rot_fullscreen,
		yoke);

  yoke[0] = plot;

  /* set up the helpshell widgets */

  helpbox = XtVaCreateManagedWidget("help_box", boxWidgetClass, helpshell,
				    NULL);

  helpdone = XtVaCreateManagedWidget("help_done", commandWidgetClass, helpbox,
				     XtNlabel, "Dismiss", NULL);

  helptextbox = XtVaCreateManagedWidget("help_textbox", asciiTextWidgetClass,
    helpbox, XtNdisplayCaret, False, XtNstring, rot_helpstring,
    XtNautoFill, True, XtNwidth, ROT_HELPWIDTH, XtNheight, ROT_HELPHEIGHT,
    XtNleftMargin, 5, XtNwrap, XawtextWrapWord, XtNscrollVertical,
    XawtextScrollWhenNeeded, XtNrightMargin, 5, XtNtopMargin, 5,
    XtNbottomMargin, 5, XtNfromVert, helpdone, NULL);
  
  XtAddCallback(helpdone, XtNcallback, (XtCallbackProc)rot_helpremove,
		helpshell);

  /* set up the zoomshell widgets */
/* fprintf(stderr," set up the zoomshell widgets\n"); */

  zoomform = XtVaCreateManagedWidget("rotform", formWidgetClass, zoomshell,
				     NULL);

  zoomlabel = XtVaCreateManagedWidget("zoomlabel", labelWidgetClass, zoomform,
				      XtNborderWidth, 0, XtNlabel,
				      "Enter new values.", NULL);

  zoomolabelx = XtVaCreateManagedWidget("olabelx", labelWidgetClass, zoomform,
    XtNborderWidth, 0, XtNlabel, "Origin X:", XtNfromVert, zoomlabel,
    XtNleft, XtChainLeft, NULL);

  sprintf(text, "%.1f", ROT_DFL_X0);
  zoomsetox = XtVaCreateManagedWidget("setox", asciiTextWidgetClass, zoomform,
    XtNeditType, XawtextEdit, XtNfromHoriz, zoomolabelx,
    XtNfromVert, zoomlabel, XtNleft, XtRubber, XtNstring, text, NULL);

  zoomolabely = XtVaCreateManagedWidget("olabely", labelWidgetClass, zoomform,
    XtNborderWidth, 0, XtNlabel, "Origin Y:", XtNfromVert, zoomolabelx,
    XtNleft, XtChainLeft, NULL);

  sprintf(text, "%.1f", ROT_DFL_Y0);
  zoomsetoy = XtVaCreateManagedWidget("setoy", asciiTextWidgetClass, zoomform,
    XtNeditType, XawtextEdit, XtNfromHoriz, zoomolabely,
    XtNfromVert, zoomolabelx, XtNleft, XtRubber, XtNstring, text, NULL);

  zoomolabelz = XtVaCreateManagedWidget("olabelz", labelWidgetClass, zoomform,
    XtNborderWidth, 0, XtNlabel, "Origin Z:", XtNfromVert, zoomolabely,
    XtNleft, XtChainLeft, NULL);

  sprintf(text, "%.1f", ROT_DFL_Z0);
  zoomsetoz = XtVaCreateManagedWidget("setoz", asciiTextWidgetClass, zoomform,
    XtNeditType, XawtextEdit, XtNfromHoriz, zoomolabelz,
    XtNfromVert, zoomolabely, XtNright, XtRubber, XtNstring, text, NULL);

  zoomrlabel = XtVaCreateManagedWidget("rlabel", labelWidgetClass, zoomform,
    XtNborderWidth, 0, XtNlabel, "Radius:  ", XtNfromVert, zoomolabelz,
    XtNleft, XtChainLeft, NULL);

  XtVaGetValues(yoke[0], XtNxmin, &minx, XtNxmax, &maxx, XtNymin, &miny,
		XtNymax, &maxy, NULL);
  sprintf(text, "%.1f\n", max2((maxx - minx), (maxy - miny)));

  zoomsetr = XtVaCreateManagedWidget("setr", asciiTextWidgetClass, zoomform,
    XtNeditType, XawtextEdit, XtNfromVert, zoomsetoz, XtNfromHoriz, zoomrlabel,
    XtNleft, XtRubber, XtNstring, text, NULL);

  zoombox = XtVaCreateManagedWidget("zoombox", boxWidgetClass, zoomform,
    XtNorientation, XtorientHorizontal, XtNhSpace, 10, XtNfromVert, zoomsetr,
    XtNborderWidth, 0, NULL);

  zoomapply = XtVaCreateManagedWidget("zoomapply", commandWidgetClass, zoombox,
				      XtNlabel, "Apply", NULL);

  zoomdone = XtVaCreateManagedWidget("zoomdone", commandWidgetClass, zoombox,
				     XtNlabel, "Done", NULL);

  yoke[6] = zoomsetox;
  yoke[7] = zoomsetoy;
  yoke[8] = zoomsetoz;
  yoke[9] = zoomsetr;

  XtAddCallback(zoomdone, XtNcallback, (XtCallbackProc)rot_zoom_remove,
		zoomshell);
  XtAddCallback(zoomapply, XtNcallback, (XtCallbackProc)rot_zoom_apply,
		yoke);

  /* set up the rotateshell widgets */
 /* fprintf(stderr," set up the rotateshell widgets\n"); */
  rotform = XtVaCreateManagedWidget("rotform", formWidgetClass, rotateshell,
				    NULL);

  rotlabel = XtVaCreateManagedWidget("rotlabel", labelWidgetClass, rotform,
    XtNborderWidth, 0, XtNlabel, "Enter rotation angles.", NULL);

  rotxlabel = XtVaCreateManagedWidget("xlabel", labelWidgetClass, rotform,
    XtNborderWidth, 0, XtNlabel, "X axis:", XtNfromVert, rotlabel,
    XtNleft, XtChainLeft, NULL);

  sprintf(text, "%.1f", ROT_DFL_XR);
  rotsetx = XtVaCreateManagedWidget("setx", asciiTextWidgetClass, rotform,
    XtNeditType, XawtextEdit, XtNfromHoriz, rotxlabel, XtNfromVert, rotlabel,
    XtNleft, XtRubber, XtNstring, text, NULL);

  rotylabel = XtVaCreateManagedWidget("ylabel", labelWidgetClass, rotform,
    XtNborderWidth, 0, XtNlabel, "Y axis:", XtNfromVert, rotxlabel,
    XtNleft, XtChainLeft, NULL);

  sprintf(text, "%.1f", ROT_DFL_YR);
  rotsety = XtVaCreateManagedWidget("sety", asciiTextWidgetClass, rotform,
    XtNeditType, XawtextEdit, XtNfromVert, rotsetx, XtNfromHoriz, rotylabel,
    XtNleft, XtRubber, XtNstring, text, NULL);

  rotzlabel = XtVaCreateManagedWidget("zlabel", labelWidgetClass, rotform,
    XtNborderWidth, 0, XtNlabel, "Z axis:", XtNfromVert, rotylabel,
    XtNleft, XtChainLeft, NULL);

  sprintf(text, "%.1f", ROT_DFL_ZR);
  rotsetz = XtVaCreateManagedWidget("setz", asciiTextWidgetClass, rotform,
    XtNeditType, XawtextEdit, XtNfromVert, rotsety, XtNfromHoriz, rotzlabel,
    XtNleft, XtRubber, XtNstring, text, NULL);

  rotbox = XtVaCreateManagedWidget("rotbox", boxWidgetClass, rotform,
    XtNorientation, XtorientHorizontal, XtNhSpace, 5, XtNfromVert, rotsetz,
    XtNborderWidth, 0, NULL);

  rotapply = XtVaCreateManagedWidget("rotapply", repeaterWidgetClass, rotbox,
    XtNlabel, "Apply", XtNshapeStyle, XmuShapeRoundedRectangle,
    XtNinitialDelay, 750, XtNrepeatDelay, 300, XtNdecay, 15, NULL);

  rotdone = XtVaCreateManagedWidget("rotdone", commandWidgetClass, rotbox,
				    XtNlabel, "Done", NULL);

  yoke[1] = rotsetx;
  yoke[2] = rotsety;
  yoke[3] = rotsetz;

  XtAddCallback(rotdone, XtNcallback, (XtCallbackProc)rot_rotate_remove,
		rotateshell);
  XtAddCallback(rotapply, XtNcallback, (XtCallbackProc)rot_rotate_apply,
		yoke);


  /* fprintf(stderr,"done with rot_main\n"); */
  
  return(topshell);
  }

/* output point data to the screen */

static void rot_output_screen(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;
  RotateWidget cw = (RotateWidget)yoke[0];
  int i, flags;
  ptset_t *pts, *container;
  float *xp, *yp, *zp;
  axes_t *axes;
  float minx, maxx, miny, maxy, xaa, yaa, zaa;

  XtVaGetValues((Widget)cw, XtNpointSet, &pts, XtNaxes, &axes, XtNxmin, &minx,
		XtNxmax, &maxx, XtNymin, &miny, XtNymax, &maxy, XtNxAngle,
		&xaa, XtNyAngle, &yaa, XtNzAngle, &zaa, XtNcontainer,
		&container, XtNsetFlags, &flags, NULL);

  puts("-- ranges --");
  printf("Xmin = %f, Xmax = %f, Ymin = %f, Ymax = %f\n", minx, maxx, miny,
	 maxy);
  if(flags & ROTMASK_ANGLES)
    printf("Xangle = %f, Yangle = %f, Zangle = %f\n", xaa, yaa, zaa);
  else
    puts("(angles not presently displayed)");

  if(flags & ROTMASK_CONTAINER)
    {
    puts("-- container line data --");

    for(i = 0, xp = container->x, yp = container->y, zp = container->z;
	i < container->n; i+=2, xp+=2, yp+=2, zp+=2)
      {
      printf("(%f, %f, %f) - (%f, %f, %f)\n", *xp, *yp, *zp, *(xp+1),
	     *(yp+1), *(zp+1));
      }

    puts("-- end of container line data --");
    }
  else puts("(container not presently displayed)");

  puts("-- axis endpoints --");
  printf("X: (%f, %f)\n", axes->xfx, axes->xfy);
  printf("Y: (%f, %f)\n", axes->yfx, axes->yfy);
  printf("Z: (%f, %f)\n", axes->zfx, axes->zfy);

  if(pts)
    {
    printf("-- %i points --\n", pts->n);
    for(i = 0, xp = pts->x, yp = pts->y, zp = pts->z; i < pts->n; i++, xp++,
	yp++, zp++)
      {
      printf("(%f, %f, %f)\n", *xp, *yp, *zp);
      }
    puts("-- end of points--");
    }

  XtVaGetValues((Widget)cw, XtNlineSet, &pts, NULL);
  if(pts)
    {
    printf("-- %i lines --\n", (pts->n / 2));
    for(i = 0, xp = pts->x, yp = pts->y, zp = pts->z; i < pts->n; i+=2,
	xp+=2, yp+=2, zp+=2)
      {
      printf("(%f, %f, %f) - (%f, %f, %f)\n", *xp, *yp, *zp, *(xp+1),
	     *(yp+1), *(zp+1));
      }
    puts("-- end of lines --");
    }
  }

/* callback function to pop up the zoom dialog box */

static void rot_zoom(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget zoomshell = (Widget)clidat;

  XtPopup(zoomshell, XtGrabNone);
  }

/* callback function to pop down the zoom dialog box */

static void rot_zoom_remove(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget zoomshell = (Widget)clidat;

  XtPopdown(zoomshell);
  }

/* callback function to apply changes made in the zoom dialog box */

static void rot_zoom_apply(Widget w, XtPointer clidat, XtPointer calldat)
  {
  float x0, y0, z0, r;
  char *text;
  Widget *yoke = (Widget *)clidat;

  XtVaGetValues(yoke[6], XtNstring, &text, NULL);
  sscanf(text, "%f", &x0);

  XtVaGetValues(yoke[7], XtNstring, &text, NULL);
  sscanf(text, "%f", &y0);

  XtVaGetValues(yoke[8], XtNstring, &text, NULL);
  sscanf(text, "%f", &z0);

  XtVaGetValues(yoke[9], XtNstring, &text, NULL);
  sscanf(text, "%f", &r);

  rot_set_zoom_resource(yoke[0], x0, y0, z0, r);
  }

/* callback function to toggle fullscreen mode */

static void rot_fullscreen(Widget w, XtPointer clidat, XtPointer calldat)
  {
  static Boolean fullflag = FALSE;
  static Position x, y;
  static Dimension width, height;
  Widget *yoke = (Widget *)clidat;

  if(fullflag == FALSE)
    {
    fullflag = TRUE;

    XtVaGetValues(yoke[10], XtNx, &x, XtNy, &y, XtNwidth, &width,
		  XtNheight, &height, NULL);
    XtVaSetValues(yoke[10], XtNx, 10, XtNy, 10, XtNwidth, 
		  WidthOfScreen(XtScreen(w)) - 20, XtNheight,
		  HeightOfScreen(XtScreen(w)) - 20, NULL);
    }
  else
    {
    XtVaSetValues(yoke[10], XtNx, x, XtNy, y, XtNwidth, width, XtNheight,
		  height, NULL);
    fullflag = FALSE;
    }
  }

/* callback function to pop up the rotate dialog box */

static void rot_rotate(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget rotateshell = (Widget)clidat;  

  XtPopup(rotateshell, XtGrabNone);
  }

/* callback function to pop down the rotate dialog box */

static void rot_rotate_remove(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget rotateshell = (Widget)clidat;

  XtPopdown(rotateshell);
  }

/* callback function to apply values in the rotate dialog box */

static void rot_rotate_apply(Widget w, XtPointer clidat, XtPointer calldat)
  {
  float xr, yr, zr;
  char *text;
  Widget *yoke = (Widget *)clidat;  

  XtVaGetValues(yoke[1], XtNstring, &text, NULL);
  sscanf(text, "%f", &xr);

  XtVaGetValues(yoke[2], XtNstring, &text, NULL);
  sscanf(text, "%f", &yr);

  XtVaGetValues(yoke[3], XtNstring, &text, NULL);
  sscanf(text, "%f", &zr);

  rot_set_rotate_resource(yoke[0], xr, yr, zr);
  }

/* callback function to reset the rotation to initial orientation */

static void rot_reset(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;
  int flags;
  char text[30];

  XtVaGetValues(yoke[0], XtNsetFlags, &flags, NULL);
  flags |= ROTMASK_RESET;
  XtVaSetValues(yoke[0], XtNsetFlags, flags, NULL);

  sprintf(text, "%.1f", ROT_DFL_XR);
  XtVaSetValues(yoke[1], XtNstring, text, NULL);

  sprintf(text, "%.1f", ROT_DFL_YR);
  XtVaSetValues(yoke[2], XtNstring, text, NULL);

  sprintf(text, "%.1f", ROT_DFL_ZR);
  XtVaSetValues(yoke[3], XtNstring, text, NULL);  

  sprintf(text, "%.1f", ROT_DFL_X0);
  XtVaSetValues(yoke[6], XtNstring, text, NULL);

  sprintf(text, "%.1f", ROT_DFL_Y0);
  XtVaSetValues(yoke[7], XtNstring, text, NULL);

  sprintf(text, "%.1f", ROT_DFL_Z0);
  XtVaSetValues(yoke[8], XtNstring, text, NULL);  

  sprintf(text, "%.1f", rot_dfl_radius);
  XtVaSetValues(yoke[9], XtNstring, text, NULL);
  }

/* callback function to pop up the help dialog box */

static void rot_help(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget helpshell = (Widget)clidat;

  XtPopup(helpshell, XtGrabNone);
  }

/* callback function to pop down the help dialog box */

static void rot_helpremove(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget helpshell = (Widget)clidat;

  XtPopdown(helpshell);
  }

/* callback function to exit everything */

void rot_exit(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;

  XtDestroyWidget(yoke[10]);
  XmapFree(yoke);
  }

/* the following are callbacks to handle the x, y, and z repeater buttons */

static void rot_rotxl(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;

  rot_set_rotate_resource(yoke[0], -ROT_DFL_INCREMENT, 0, 0);
  }

static void rot_rotxr(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;

  rot_set_rotate_resource(yoke[0], ROT_DFL_INCREMENT, 0, 0);
  }

static void rot_rotyl(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;

  rot_set_rotate_resource(yoke[0], 0, -ROT_DFL_INCREMENT, 0);
  }

static void rot_rotyr(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;

  rot_set_rotate_resource(yoke[0], 0, ROT_DFL_INCREMENT, 0);
  }

static void rot_rotzl(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;

  rot_set_rotate_resource(yoke[0], 0, 0, -ROT_DFL_INCREMENT);
  }

static void rot_rotzr(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;

  rot_set_rotate_resource(yoke[0], 0, 0, ROT_DFL_INCREMENT);
  }

/*
 *  Xt apparently doesn't like floats passed in XtVaSetValues(), so we
 *  have to make do with this kludge. (X Toolkit Programming Intrinsics
 *  Manual (O'Reilly & Associates) vol. 4, p. 46)
 */

static void rot_set_zoom_resource(Widget w, float x0, float y0, float z0,
				  float r)
  {
  Arg args[8];
  union 
    {
    int iv;
    float fv;
    } vv;

  vv.fv = x0;
  XtSetArg(args[0], XtNorigX, vv.iv);
  vv.fv = y0;
  XtSetArg(args[1], XtNorigY, vv.iv);
  vv.fv = z0;
  XtSetArg(args[2], XtNorigZ, vv.iv);
  vv.fv = r;
  XtSetArg(args[3], XtNradius, vv.iv);

  XtSetArg(args[4], XtNrotX, 0);
  XtSetArg(args[5], XtNrotY, 0);
  XtSetArg(args[6], XtNrotZ, 0);

  XtSetValues(w, args, 7);
  }

static void rot_set_rotate_resource(Widget w, float xr, float yr, float zr)
  {
  Arg args[4];
  union 
    {
    int iv;
    float fv;
    } vv;

  vv.fv = xr;
  XtSetArg(args[0], XtNrotX, vv.iv);
  vv.fv = yr;
  XtSetArg(args[1], XtNrotY, vv.iv);
  vv.fv = zr;
  XtSetArg(args[2], XtNrotZ, vv.iv);

  XtSetValues(w, args, 3);
  }

/* callback function to handle fancy plot */

static void rot_fancy(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;
  int flags;

  XtVaGetValues(yoke[0], XtNsetFlags, &flags, NULL);
  flags |= ROTMASK_FANCY;
  XtVaSetValues(yoke[0], XtNsetFlags, flags, NULL);
  }

/* callback function to display orientation angles */

static void rot_show_angles(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;
  int flags;

  XtVaGetValues(yoke[0], XtNsetFlags, &flags, NULL);

  if(flags & ROTMASK_ANGLES)
    {
    flags &= ~ROTMASK_ANGLES;
    XtVaSetValues(m2_angles, XtNlabel, "Show Angles", NULL);
    }
  else
    {
    flags |= ROTMASK_ANGLES;
    XtVaSetValues(m2_angles, XtNlabel, "Hide Angles", NULL);
    }

  XtVaSetValues(yoke[0], XtNsetFlags, flags, NULL);
  }

/* callback function to toggle display of container */

static void rot_container(Widget w, XtPointer clidat, XtPointer calldat)
  {
  Widget *yoke = (Widget *)clidat;
  int flags;

  XtVaGetValues(yoke[0], XtNsetFlags, &flags, NULL);

  if(flags & ROTMASK_CONTAINER)
    {
    flags &= ~ROTMASK_CONTAINER;
    XtVaSetValues(m2_container, XtNlabel, "Show Container", NULL);
    }
  else
    {
    flags |= ROTMASK_CONTAINER;
    XtVaSetValues(m2_container, XtNlabel, "Hide Container", NULL);
    }

  XtVaSetValues(yoke[0], XtNsetFlags, flags, NULL);
  }

/* end of source file */
