/*
 * ---------------------------------------------------------------------------
 * rotwidget.c - RotateWidget source
 *
 * Mark Lindner - 2/10/96
 * ---------------------------------------------------------------------------
 */

/* feature switches */

#define _POSIX_SOURCE 1

/* --- system headers --- */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>

/* --- local headers --- */

#include "geotouch.h"
#include "rotwidgetP.h"

/*---  macros --- */

#define DFL_WIDTH  700
#define DFL_HEIGHT 400
#define DFL_FONT "9x15"
#define DFL_RANGE 40
#define DFL_RADIUS 10.0

#define MIN_WIDTH 200
#define MIN_HEIGHT 200

#define HELPSTRING ""

#define AXISLEN 4
#define SPHERERAD 4


 static float rotw_vangle(float, float, float, float, float, float);
 static void rotw_recalc_matrix(RotateWidget);
  static void rotw_init_axes(RotateWidget);
  static void rotw_init_container(RotateWidget);
  static void rotw_translate_origin(RotateWidget);

 static void rotw_recalc_axes(RotateWidget);
  static void rotw_recalc_points(RotateWidget);
  static void rotw_calc_angles(RotateWidget);
static void rotw_project_point(RotateWidget, float, float, float,short *, short *);


/* --- global variables --- */

const float dfl_x0 = 0.0, dfl_y0 = 0.0, dfl_z0 = 0.0, dfl_radius = 10.0,
  dfl_zero = 0.0, dfl_max = 10, dfl_min = -10;
static Boolean redraw = False, fancy = False, container = False,
  angles = False;

static XtResource resources[] =
  {
    {
    XtNcallback, 
    XtCCallback, 
    XtRCallback, 
    sizeof(XtPointer),
    XtOffsetOf(RotateRec, rotate.callback), 
    XtRCallback, 
    NULL,
    },
    /* custom resources */
    {
    XtNfontName,
    XtCFontName,
    XtRString,
    sizeof(XtPointer),
    XtOffsetOf(RotateRec, rotate.font_name), 
    XtRString, 
    (XtPointer)DFL_FONT
    },
    {
    XtNsetFlags,
    XtCSetFlags,
    XtRInt,
    sizeof(int),
    XtOffsetOf(RotateRec, rotate.flags),
    XtRImmediate,
    (XtPointer)&dfl_zero
    },
    {
    XtNrotX,
    XtCRotX,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.xr),
    XtRFloat,
    (XtPointer)&dfl_zero
    },
    {
    XtNrotY,
    XtCRotY,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.yr),
    XtRFloat,
    (XtPointer)&dfl_zero
    },
    {
    XtNrotZ,
    XtCRotZ,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.zr),
    XtRFloat,
    (XtPointer)&dfl_zero
    },
    {
    XtNorigX,
    XtCOrigX,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.x0),
    XtRImmediate,
    (XtPointer)&dfl_x0
    },
    {
    XtNorigY,
    XtCOrigY,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.y0),
    XtRImmediate,
    (XtPointer)&dfl_y0
    },
    {
    XtNorigZ,
    XtCOrigZ,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.z0),
    XtRImmediate,
    (XtPointer)&dfl_z0
    },
    {
    XtNradius,
    XtCRadius,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.radius),
    XtRImmediate,
    (XtPointer)&dfl_radius
    },
    {
    XtNpointSet,
    XtCPointSet,
    XtRPointer,
    sizeof(XtPointer),
    XtOffsetOf(RotateRec, rotate.pointset),
    XtRImmediate,
    NULL
    },
    {
    XtNlineSet,
    XtCLineSet,
    XtRPointer,
    sizeof(XtPointer),
    XtOffsetOf(RotateRec, rotate.lineset),
    XtRImmediate,
    NULL
    },
    {
    XtNcontainer,
    XtCContainer,
    XtRPointer,
    sizeof(XtPointer),
    XtOffsetOf(RotateRec, rotate.ctr),
    XtRImmediate,
    NULL
    },
    {
    XtNaxes,
    XtCAxes,
    XtRPointer,
    sizeof(XtPointer),
    XtOffsetOf(RotateRec, rotate.axes),
    XtRImmediate,
    NULL
    },
    {
    XtNxmin,
    XtCXmin,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.minx),
    XtRFloat,
    (XtPointer)&dfl_min
    },
    {
    XtNxmax,
    XtCXmax,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.maxx),
    XtRFloat,
    (XtPointer)&dfl_max
    },
    {
    XtNymin,
    XtCYmin,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.miny),
    XtRFloat,
    (XtPointer)&dfl_min
    },
    {
    XtNymax,
    XtCYmax,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.maxy),
    XtRFloat,
    (XtPointer)&dfl_max
    },
    {
    XtNxAngle,
    XtCXAngle,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.xaa),
    XtRFloat,
    (XtPointer)&dfl_zero
    },
    {
    XtNyAngle,
    XtCYAngle,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.yaa),
    XtRFloat,
    (XtPointer)&dfl_zero
    },
    {
    XtNzAngle,
    XtCZAngle,
    XtRFloat,
    sizeof(float),
    XtOffsetOf(RotateRec, rotate.zaa),
    XtRFloat,
    (XtPointer)&dfl_zero
    },
  };

/* action table */

static XtActionsRec actions[] =
  {
  };

/* default translations */

static char defaultTranslations[] =
  "";

/* method handlers */

static void rotw_initialize(Widget, Widget, ArgList, Cardinal *);
static void rotw_expose(/* Widget, XExposeEvent *, Region */);
static Boolean rotw_set_values(Widget, Widget, Widget, ArgList, Cardinal *);
static void rotw_resize(Widget );
static XtGeometryResult rotw_query_geometry(Widget, XtWidgetGeometry *,
  XtWidgetGeometry *);
static void rotw_destroy(Widget );

RotateClassRec rotateClassRec =
  {
    {
    /* core_class fields */
    /* superclass	  	 */ (WidgetClass)&coreClassRec,
    /* class_name	  	 */ "Rotate",
    /* widget_size	  	 */ sizeof(RotateRec),
    /* class_initialize   	 */ NULL,
    /* class_part_initialize	 */ NULL,
    /* class_inited       	 */ FALSE,
    /* initialize	  	 */ rotw_initialize,
    /* initialize_hook		 */ NULL,
    /* realize		  	 */ XtInheritRealize,
    /* actions		  	 */ actions,
    /* num_actions	  	 */ XtNumber(actions),
    /* resources	  	 */ resources,
    /* num_resources	  	 */ XtNumber(resources),
    /* xrm_class	  	 */ NULLQUARK,
    /* compress_motion	  	 */ TRUE,
    /* compress_exposure  	 */ XtExposeCompressMultiple,
    /* compress_enterleave	 */ TRUE,
    /* visible_interest	  	 */ FALSE,
    /* destroy		  	 */ rotw_destroy,
    /* resize		  	 */ rotw_resize,
    /* expose		  	 */ rotw_expose,
    /* set_values	  	 */ rotw_set_values,
    /* set_values_hook		 */ NULL,
    /* set_values_almost	 */ XtInheritSetValuesAlmost,
    /* get_values_hook		 */ NULL,
    /* accept_focus	 	 */ NULL,
    /* version			 */ XtVersion,
    /* callback_private   	 */ NULL,
    /* tm_table		   	 */ defaultTranslations,
    /* query_geometry		 */ rotw_query_geometry,
    /* display_accelerator       */ XtInheritDisplayAccelerator,
    /* extension                 */ NULL
    },
    {
    /* dummy_field               */ 0
    }
  };

WidgetClass rotateWidgetClass = (WidgetClass)&rotateClassRec;

/* --------------------- method handlers --------------------- */

/*
 *  rotw_initialize() : handle initialize method for RotateWidget
 */
/*
 *  rotw_translate_origin(): apply an origin translation to the points
 */

static void rotw_translate_origin(RotateWidget w)
  {
  RotateWidget cw = (RotateWidget)w;
  float *xc, *yc, *zc;
  int i;

  /* calculate the points for pointset */

  if(cw->rotate.pointsetf)
    {
    for(i = 0, xc = cw->rotate.lpointset.x, yc = cw->rotate.lpointset.y,
        zc = cw->rotate.lpointset.z;
        i < cw->rotate.lpointset.n; i++, xc++, yc++, zc++)
      {
      *xc += cw->rotate.x0;
      *yc += cw->rotate.y0;
      *zc += cw->rotate.z0;
      }
    }

  /* calculate the points for the lineset */

  if(cw->rotate.linesetf)
    {
    for(i = 0, xc = cw->rotate.llineset.x, yc = cw->rotate.llineset.y,
        zc = cw->rotate.llineset.z;
        i < cw->rotate.llineset.n; i++, xc++, yc++, zc++)
      {
      *xc += cw->rotate.x0;
      *yc += cw->rotate.y0;
      *zc += cw->rotate.z0;
      }
    }

  /* calculate points for container */

  for(i = 0; i < 24; i++)
      {
      cw->rotate.container.x[i] += cw->rotate.x0; 
      cw->rotate.container.y[i] += cw->rotate.y0; 
      cw->rotate.container.z[i] += cw->rotate.z0; 
      }
  }

/*
 *  rotw_calc_angles(): calculate the angles for the "show angles" button
 */

static void rotw_calc_angles(RotateWidget w)
  {
  RotateWidget cw = (RotateWidget)w;
  char text[80];

 
  /* compute angles between abs-z and x, y, and z */

  cw->rotate.xaa = rotw_vangle(0, 0, 1, cw->rotate.axes->xfx,
			       cw->rotate.axes->xfy, cw->rotate.axes->xfz);
  cw->rotate.yaa = rotw_vangle(0, 0, 1, cw->rotate.axes->yfx,
			       cw->rotate.axes->yfy, cw->rotate.axes->yfz);
  cw->rotate.zaa = rotw_vangle(0, 0, 1, cw->rotate.axes->zfx,
			       cw->rotate.axes->zfy, cw->rotate.axes->zfz);

  sprintf(text, "(x: %.1f, y: %.1f, z: %.1f)", rad2deg(cw->rotate.xaa),
	  rad2deg(cw->rotate.yaa), rad2deg(cw->rotate.zaa));
  XDrawString(XtDisplay(cw), XtWindow(cw), cw->rotate.draw_gc,
	      5, (cw->core.height - 5), text, strlen(text));
  }



static void rotw_initialize(Widget treq, Widget tnew, ArgList args,
  Cardinal *num_args)
  {
  int sizemem, i;
  float *xp, *yp, *zp;
  RotateWidget new = (RotateWidget)tnew;
  XGCValues values;
  XtGCMask mask = (GCForeground | GCBackground | GCFont | GCLineWidth);

 
  /* fonts */

  new->rotate.font_info = XLoadQueryFont(XtDisplay(new),
					 new->rotate.font_name);

  /* flags */

  new->rotate.flags = 0;

  new->rotate.pointsetf = False;
  new->rotate.linesetf = False;

  new->rotate.minx = new->rotate.miny = new->rotate.minz = HUGE_VAL;
  new->rotate.maxx = new->rotate.maxy = new->rotate.maxz = -HUGE_VAL;

  new->rotate.ctr = &(new->rotate.container);

  /* do we have a pointset? */

  if(new->rotate.pointset)
    {
    if(new->rotate.pointset->n > 0)
      {
      sizemem = new->rotate.pointset->n * sizeof(float);
      new->rotate.pointsetf = True;

      /* make private copy of original data */

      new->rotate.ppointset.n = new->rotate.pointset->n;
      new->rotate.ppointset.x = XmapNewArray(float, new->rotate.pointset->n);
      new->rotate.ppointset.y = XmapNewArray(float, new->rotate.pointset->n);
      new->rotate.ppointset.z = XmapNewArray(float, new->rotate.pointset->n);
      memcpy((void *)new->rotate.ppointset.x, (void *)new->rotate.pointset->x,
	     (size_t)sizemem);
      memcpy((void *)new->rotate.ppointset.y, (void *)new->rotate.pointset->y,
	     (size_t)sizemem);
      memcpy((void *)new->rotate.ppointset.z, (void *)new->rotate.pointset->z,
	     (size_t)sizemem);

      new->rotate.lpointset.n = new->rotate.pointset->n;
      new->rotate.lpointset.x = XmapNewArray(float, new->rotate.pointset->n);
      new->rotate.lpointset.y = XmapNewArray(float, new->rotate.pointset->n);
      new->rotate.lpointset.z = XmapNewArray(float, new->rotate.pointset->n);
      memcpy((void *)new->rotate.lpointset.x, (void *)new->rotate.ppointset.x,
	     (size_t)sizemem);
      memcpy((void *)new->rotate.lpointset.y, (void *)new->rotate.ppointset.y,
	     (size_t)sizemem);
      memcpy((void *)new->rotate.lpointset.z, (void *)new->rotate.ppointset.z,
	     (size_t)sizemem);

      /* dump original & redirect pointer */

      XmapFree(new->rotate.pointset->x);
      XmapFree(new->rotate.pointset->y);
      XmapFree(new->rotate.pointset->z);

      new->rotate.pointset = &(new->rotate.lpointset);

      new->rotate.points = XmapNewArray(XPoint, new->rotate.lpointset.n);

      /* calculate min and max of these points */

      for(i = 0, xp = new->rotate.ppointset.x, yp = new->rotate.ppointset.y,
	  zp = new->rotate.ppointset.z; i < new->rotate.ppointset.n;
	  i++, xp++, yp++, zp++)
	{
	if(*xp < new->rotate.minx) new->rotate.minx = *xp;
	else if(*xp > new->rotate.maxx) new->rotate.maxx = *xp;

	if(*yp < new->rotate.miny) new->rotate.miny = *yp;
	else if(*yp > new->rotate.maxy) new->rotate.maxy = *yp;

	if(*zp < new->rotate.minz) new->rotate.minz = *zp;
	else if(*zp > new->rotate.maxz) new->rotate.maxz = *zp;	
	}
      }
    }

  /* do we have a lineset? */

  if(new->rotate.lineset)
    {
    if(new->rotate.lineset->n > 0)
      {
      if(new->rotate.lineset->n % 2)
	{
	fprintf(stderr,
		"RotateWidget warning: odd # of points for wireframe.\n"
		"Chopping last point.\n");
	(new->rotate.lineset->n)--;
	}

      sizemem = (new->rotate.lineset->n * sizeof(float));
      new->rotate.linesetf = True;

      /* make private copy of original data */

      new->rotate.plineset.n = new->rotate.lineset->n;
      new->rotate.plineset.x = XmapNewArray(float, new->rotate.lineset->n);
      new->rotate.plineset.y = XmapNewArray(float, new->rotate.lineset->n);
      new->rotate.plineset.z = XmapNewArray(float, new->rotate.lineset->n);
      memcpy((void *)new->rotate.plineset.x,
	     (void *)new->rotate.lineset->x, (size_t)sizemem);
      memcpy((void *)new->rotate.plineset.y,
	     (void *)new->rotate.lineset->y, (size_t)sizemem);
      memcpy((void *)new->rotate.plineset.z,
	     (void *)new->rotate.lineset->z, (size_t)sizemem);

      new->rotate.llineset.n = new->rotate.plineset.n;
      new->rotate.llineset.x = XmapNewArray(float, new->rotate.lineset->n);
      new->rotate.llineset.y = XmapNewArray(float, new->rotate.lineset->n);
      new->rotate.llineset.z = XmapNewArray(float, new->rotate.lineset->n);
      memcpy((void *)new->rotate.llineset.x, (void *)new->rotate.plineset.x,
	     (size_t)sizemem);
      memcpy((void *)new->rotate.llineset.y, (void *)new->rotate.plineset.y,
	     (size_t)sizemem);
      memcpy((void *)new->rotate.llineset.z, (void *)new->rotate.plineset.z,
	     (size_t)sizemem);

      /* dump original & redirect pointer */

      XmapFree(new->rotate.lineset->x);
      XmapFree(new->rotate.lineset->y);
      XmapFree(new->rotate.lineset->z);
      /* XmapFree(new->rotate.lineset); */

      new->rotate.lineset = &(new->rotate.llineset);

      new->rotate.lines = XmapNewArray(XPoint, new->rotate.llineset.n);

      /* calculate min and max of these points */

      for(i = 0, xp = new->rotate.plineset.x, yp = new->rotate.plineset.y,
	  zp = new->rotate.plineset.z; i < new->rotate.plineset.n;
	  i++, xp++, yp++, zp++)
	{
	if(*xp < new->rotate.minx) new->rotate.minx = *xp;
	else if(*xp > new->rotate.maxx) new->rotate.maxx = *xp;

	if(*yp < new->rotate.miny) new->rotate.miny = *yp;
	else if(*yp > new->rotate.maxy) new->rotate.maxy = *yp;

	if(*zp < new->rotate.minz) new->rotate.minz = *zp;
	else if(*zp > new->rotate.maxz) new->rotate.maxz = *zp;	
	}
      }
    }

  /* check if ranges are sensible, and scale them */

  if(new->rotate.minx <= (-HUGE_VAL / 2)) new->rotate.minx = -DFL_RADIUS;
  if(new->rotate.maxx >= (HUGE_VAL / 2)) new->rotate.maxx = DFL_RADIUS;
  if(new->rotate.miny <= (-HUGE_VAL / 2)) new->rotate.miny = -DFL_RADIUS;
  if(new->rotate.maxy >= (HUGE_VAL / 2)) new->rotate.maxy = DFL_RADIUS;
  if(new->rotate.minz <= (-HUGE_VAL / 2)) new->rotate.minz = -DFL_RADIUS;
  if(new->rotate.maxz >= (HUGE_VAL / 2)) new->rotate.maxz = DFL_RADIUS;

  new->rotate.minx *= 1.05;
  new->rotate.maxx *= 1.05;
  new->rotate.miny *= 1.05;
  new->rotate.maxy *= 1.05;

  new->rotate.axes = XmapNewElement(axes_t);

  /* check for out-of-range resource values */

  if(new->core.width < MIN_WIDTH) new->core.width = DFL_WIDTH;
  if(new->core.height < MIN_HEIGHT) new->core.height = DFL_HEIGHT;

  /* check rotation angles */

  if(fabs(new->rotate.xr) > 360) new->rotate.xr = 0.0;
  new->rotate.xrr = deg2rad(new->rotate.xr);
  if(fabs(new->rotate.yr) > 360) new->rotate.yr = 0.0;
  new->rotate.yrr = deg2rad(new->rotate.yr);
  if(fabs(new->rotate.zr) > 360) new->rotate.zr = 0.0;
  new->rotate.zrr = deg2rad(new->rotate.zr);

  /* compute the radius and origin */

  new->rotate.radius = max2(max2(fabs(new->rotate.maxx - new->rotate.minx),
				 fabs(new->rotate.maxy - new->rotate.miny)),
			    fabs(new->rotate.maxz - new->rotate.minz));

  if(new->rotate.radius == 0) new->rotate.radius = dfl_radius;

  new->rotate.x0 = -(new->rotate.maxx + new->rotate.minx) / 2;
  new->rotate.y0 = -(new->rotate.maxy + new->rotate.miny) / 2;
  new->rotate.z0 = -(new->rotate.maxz + new->rotate.minz) / 2;

  /* get drawing GC */

  values.foreground  = 1;
  values.background  = 0;
  values.dashes      = 1;
  values.dash_offset = 0;
  values.line_style  = LineSolid;
  values.font        = new->rotate.font_info->fid;
  values.line_width  = 1;


/*  new->rotate.draw_gc = XCreateGC(XtDisplay(new), ?, mask, &values); */

  new->rotate.draw_gc = XtGetGC((Widget)new, mask, &values);

  /* initialize all the housekeeping data */

  rotw_recalc_matrix(new);
  rotw_init_axes(new);

  new->rotate.container.x = XmapNewArray(float, 24);
  new->rotate.container.y = XmapNewArray(float, 24);
  new->rotate.container.z = XmapNewArray(float, 24);
  rotw_init_container(new);

  rotw_translate_origin(new);
  }

/*
 *  rotw_expose() : handle expose method for RotateWidget
 */

static void rotw_expose(Widget w, XExposeEvent *event, Region region)
  {
  static Boolean init = True;
  RotateWidget cw = (RotateWidget)w;
  int i;
  short x1, y1, x2, y2;
  XPoint *pt;
  axes_t *axes;

 			

  if(!XtIsRealized((Widget)cw)) return;

  /* if init is true, then this is our first time, so initialize */

  if(init)
    {
    rotw_recalc_points(cw);
    rotw_recalc_axes(cw);
    init = False;
    }

  /* if a redraw (a rotation advance) is necessary, do it */

  else if(redraw && !fancy)
    {
    rotw_recalc_points(cw);
    rotw_recalc_axes(cw);
    redraw = False;
    }

  /* now selectively draw the requested data */
 XSetClipMask(XtDisplay(cw), cw->rotate.draw_gc, None);
  XSetFunction(XtDisplay(cw),  cw->rotate.draw_gc, GXand);

  /* draw lines */
 /*  fprintf(stderr, "n=%d\n", cw->rotate.llineset.n); */


  if(cw->rotate.linesetf)
  {
     /* fprintf(stderr, "ROT lines\n"); */
     for(i = 0, pt = cw->rotate.lines; i < cw->rotate.llineset.n;
	 i+=2, pt+=2)
     {
	XDrawLine(XtDisplay(cw), XtWindow(cw), cw->rotate.draw_gc,
		  pt->x, pt->y, (pt+1)->x, (pt+1)->y);
	
	/* fprintf(stderr, "ROT lines\n"); */
	/* XFlush(XtDisplay(cw)); */

     }
  }

  /* draw points */

  if(cw->rotate.pointsetf)
    {
    /* do the fancy plot? */

    if(fancy)
      {
      fancy = False;
      for(i = 0, pt = cw->rotate.points; i < cw->rotate.lpointset.n;
	  i++, pt++)
	{
	XSetForeground(XtDisplay(cw), cw->rotate.draw_gc,
		       cw->rotate.pixelvals[XmapAppColor(3)]); 

	XFillArc(XtDisplay(cw), XtWindow(cw), cw->rotate.draw_gc,
		 (pt->x - SPHERERAD), (pt->y - SPHERERAD), (SPHERERAD * 2),
		 (SPHERERAD * 2), 0, (360 * 64));
	}
      }
    else
      {
      for(i = 0, pt = cw->rotate.points; i < cw->rotate.lpointset.n;
	  i++, pt++)
      XDrawRectangle(XtDisplay(cw), XtWindow(cw), cw->rotate.draw_gc,
		     pt->x, pt->y, 1, 1);
      }
    }

  /* draw axes */

  axes = cw->rotate.axes;

  XDrawLine(XtDisplay(cw), XtWindow(cw), cw->rotate.draw_gc,
	    axes->ox, axes->oy, axes->zax, axes->zay);
  XDrawLine(XtDisplay(cw), XtWindow(cw), cw->rotate.draw_gc,
	    axes->ox, axes->oy, axes->yax, axes->yay);
  XDrawLine(XtDisplay(cw), XtWindow(cw), cw->rotate.draw_gc,
	    axes->ox, axes->oy, axes->xax, axes->xay);
  XDrawString(XtDisplay(cw), XtWindow(cw), cw->rotate.draw_gc,
	      (axes->zax)+5, (axes->zay)+5, "z", 1);
  XDrawString(XtDisplay(cw), XtWindow(cw), cw->rotate.draw_gc,
	      (axes->yax)+5, (axes->yay)+5, "y", 1);
  XDrawString(XtDisplay(cw), XtWindow(cw), cw->rotate.draw_gc,
	      (axes->xax)+5, (axes->xay)+5, "x", 1);

  /* do they want to see the angles? */

  if(angles) rotw_calc_angles(cw);

  /* do they want to see the container? */

  if(container)
    {
    /* box in the data */
    for(i = 0; i < 24; i+=2)
      {
      rotw_project_point(cw, cw->rotate.container.x[i],
			 cw->rotate.container.y[i],
			 cw->rotate.container.z[i], &x1, &y1);
      rotw_project_point(cw, cw->rotate.container.x[i+1],
			 cw->rotate.container.y[i+1],
			 cw->rotate.container.z[i+1], &x2, &y2);

      XDrawLine(XtDisplay(cw), XtWindow(cw), cw->rotate.draw_gc, x1, y1,
		x2, y2);
      }
    }


  /* fprintf(stderr, "done rotwidget-ting....\n"); */
  XSetFunction(XtDisplay(cw),  cw->rotate.draw_gc, GXcopy);

  }

/*
 *  rotw_set_values() : handle set_values method for RotateWidget
 */

static Boolean rotw_set_values(Widget current, Widget request, Widget super,
  ArgList args, Cardinal *num_args)
  {
  RotateWidget curcw = (RotateWidget)current;
  RotateWidget newcw = (RotateWidget)super;
  Boolean refresh = False, recalc_matrix = False;
  int sizemem;

 
  /* check if read-only resources have been changed */

  if(newcw->rotate.pointset != curcw->rotate.pointset)
    fprintf(stderr, "Rotwidget warning: XtNpointSet resource is read-only.\n"),
    newcw->rotate.pointset = curcw->rotate.pointset;

  if(newcw->rotate.lineset != curcw->rotate.lineset)
    fprintf(stderr, "Rotwidget warning: XtNlineSet resource is read-only.\n"),
    newcw->rotate.lineset = curcw->rotate.lineset;

  if(newcw->rotate.axes != curcw->rotate.axes)
    fprintf(stderr, "Rotwidget warning: XtNaxes resource is read-only.\n"),
    newcw->rotate.axes = curcw->rotate.axes;

  /* make sure radius is positive */

  if(curcw->rotate.radius != newcw->rotate.radius)
    {
    if(newcw->rotate.radius <= 0) newcw->rotate.radius = dfl_radius;
    newcw->rotate.x_scl = (newcw->core.width / (2 * newcw->rotate.radius));
    newcw->rotate.y_scl = (newcw->core.height / (2 * newcw->rotate.radius));
    redraw = True;
    refresh = True;
    }

  /* check if geometry has changed, and if so, recalculate scale factors */

  if(curcw->core.width != newcw->core.width)
    {
    if(newcw->core.width <= 0) newcw->core.width = DFL_WIDTH;
    newcw->rotate.x_scl = (newcw->core.width / (2 * newcw->rotate.radius));
    }

  if(curcw->core.height != newcw->core.height)
    {
    if(newcw->core.height <= 0) newcw->core.height = DFL_HEIGHT;
    newcw->rotate.y_scl = (newcw->core.height / (2 * newcw->rotate.radius));
    }

  /* have special flags been changed? */

  if(newcw->rotate.flags & ROTMASK_FANCY)
    {
    newcw->rotate.flags &= ~ROTMASK_FANCY;
    fancy = True;
    }

  if(newcw->rotate.flags & ROTMASK_RESET)
    {
    newcw->rotate.flags &= ~ROTMASK_RESET;
    
    if(curcw->rotate.pointsetf)
      {
      sizemem = (curcw->rotate.lpointset.n * sizeof(float));
      
      memcpy((void *)newcw->rotate.lpointset.x,
	     (void *)newcw->rotate.ppointset.x, (size_t)sizemem);
      memcpy((void *)newcw->rotate.lpointset.y,
	     (void *)newcw->rotate.ppointset.y, (size_t)sizemem);
      memcpy((void *)newcw->rotate.lpointset.z,
	     (void *)newcw->rotate.ppointset.z, (size_t)sizemem);
      }

    if(curcw->rotate.linesetf)
      {
      sizemem = (curcw->rotate.llineset.n * sizeof(float));

      memcpy((void *)newcw->rotate.llineset.x,
	     (void *)newcw->rotate.plineset.x, (size_t)sizemem);
      memcpy((void *)newcw->rotate.llineset.y,
	     (void *)newcw->rotate.plineset.y, (size_t)sizemem);
      memcpy((void *)newcw->rotate.llineset.z,
	     (void *)newcw->rotate.plineset.z, (size_t)sizemem);
      }
    
    newcw->rotate.xr = 0, newcw->rotate.yr = 0, newcw->rotate.zr = 0;
    newcw->rotate.xrr = 0, newcw->rotate.yrr = 0, newcw->rotate.zrr = 0;
      
    newcw->rotate.x0 = -(newcw->rotate.maxx + newcw->rotate.minx) / 2;
    newcw->rotate.y0 = -(newcw->rotate.maxy + newcw->rotate.miny) / 2;
    newcw->rotate.z0 = -(newcw->rotate.maxz + newcw->rotate.minz) / 2;

    newcw->rotate.x_scl = (newcw->core.width / (2 * newcw->rotate.radius));
    newcw->rotate.y_scl = (newcw->core.height / (2 * newcw->rotate.radius));
    
    rotw_recalc_matrix(newcw);
    rotw_init_axes(newcw);
    rotw_init_container(newcw);
    rotw_translate_origin(newcw);

    redraw = True;
    }

  if(newcw->rotate.flags != curcw->rotate.flags)
    {
    angles = (newcw->rotate.flags & ROTMASK_ANGLES) ? True : False;
    container = (newcw->rotate.flags & ROTMASK_CONTAINER) ? True : False;
    
    refresh = True;
    }

  else
    {

/* check if origin has changed and is acceptable */

    if((curcw->rotate.x0 != newcw->rotate.x0) ||
       (curcw->rotate.y0 != newcw->rotate.y0) ||
       (curcw->rotate.z0 != newcw->rotate.z0))
      rotw_translate_origin(newcw);

    /* check if rotation angle values have changed & are acceptable */

    if(curcw->rotate.xr != newcw->rotate.xr)
      {
      if(fabs(newcw->rotate.xr) > 360.0) newcw->rotate.xr = 0.0;
      newcw->rotate.xrr = deg2rad(newcw->rotate.xr);
      recalc_matrix = True;
      }

    if(curcw->rotate.yr != newcw->rotate.yr)
      {
      if(fabs(newcw->rotate.yr) > 360.0) newcw->rotate.yr = 0.0;
      newcw->rotate.yrr = deg2rad(newcw->rotate.yr);
      recalc_matrix = True;
      }

    if(curcw->rotate.zr != newcw->rotate.zr)
      {
      if(fabs(newcw->rotate.zr) > 360.0) newcw->rotate.zr = 0.0;
      recalc_matrix = True;
      newcw->rotate.zrr = deg2rad(newcw->rotate.zr);
      }

    redraw = True, refresh = True;
    if(recalc_matrix == True) rotw_recalc_matrix(newcw);
    }
  
  return(refresh);
  }

/*
 *  rotw_resize() : handle resize method for RotateWidget
 */

static void rotw_resize(Widget w)
  {
  redraw = True;
  }

/*
 *  rotw_query_geometry() : handle query_geometry method for RotateWidget
 */

static XtGeometryResult rotw_query_geometry(Widget w,
  XtWidgetGeometry *proposed, XtWidgetGeometry *answer)
  {
  RotateWidget cw = (RotateWidget)w;

  if(proposed->width == cw->core.width && proposed->height == cw->core.height)
    return(XtGeometryNo);

  if(proposed->width == proposed->height)
    return(XtGeometryYes);

  if(proposed->width > proposed->height)
    answer->width = proposed->height, answer->request_mode = CWWidth;
  else
    answer->height = proposed->width, answer->request_mode = CWHeight;
  return(XtGeometryAlmost);
  }

/*
 *  rotw_destroy() : handle destroy method for RotateWidget
 */

static void rotw_destroy(Widget w)
  {
  RotateWidget cw = (RotateWidget)w;

  /* release our private memory blocks */

  if(cw->rotate.pointsetf)
    {
    XmapFree(cw->rotate.lpointset.x);
    XmapFree(cw->rotate.lpointset.y);
    XmapFree(cw->rotate.lpointset.z);
    XmapFree(cw->rotate.points);

    XmapFree(cw->rotate.ppointset.x);
    XmapFree(cw->rotate.ppointset.y);
    XmapFree(cw->rotate.ppointset.z);
    }

  if(cw->rotate.linesetf)
    {
    XmapFree(cw->rotate.llineset.x);
    XmapFree(cw->rotate.llineset.y);
    XmapFree(cw->rotate.llineset.z);
    XmapFree(cw->rotate.lines);

    XmapFree(cw->rotate.plineset.x);
    XmapFree(cw->rotate.plineset.y);
    XmapFree(cw->rotate.plineset.z);
    }

  XmapFree(cw->rotate.container.x);
  XmapFree(cw->rotate.container.y);
  XmapFree(cw->rotate.container.z);

  XmapFree(cw->rotate.axes);
  }

/* --------------------- point translation functions --------------------- */

/*
 *  rotw_project_point(): project the 3D user-coordinate point (x,y,z) to
 *  the 2D screen-coordinate point (*xp, *yp)
 */

static void rotw_project_point(RotateWidget w, float x, float y, float z,
				   short *xp, short *yp)
  {

  *xp = (w->core.width / 2) + (x * w->rotate.x_scl);
  *yp = (w->core.height / 2) - (y * w->rotate.y_scl);
  }

/*
 *  rotw_rotate_point(): rotate the point by the angles specified in the
 *  widget's resources
 */

static void rotw_rotate_point(RotateWidget w, float x, float y, float z,
			      float *xp, float *yp, float *zp)
  {

  *xp = ((x * w->rotate.matrix[0][0]) +
	 (y * w->rotate.matrix[0][1]) +
	 (z * w->rotate.matrix[0][2]));
  *yp = ((x * w->rotate.matrix[1][0]) +
	 (y * w->rotate.matrix[1][1]) +
	 (z * w->rotate.matrix[1][2]));
  *zp = ((x * w->rotate.matrix[2][0]) +
	 (y * w->rotate.matrix[2][1]) +
	 (z * w->rotate.matrix[2][2]));
  }

/*
 *  rotw_recalc_matrix(): calculate values for rotation matrix
 */

static void rotw_recalc_matrix(RotateWidget w)
  {
  float sin_r, sin_s, sin_t, cos_r, cos_s, cos_t;

  /* compute the sines and cosines once to save time */

  sin_r = (float)sin(w->rotate.xrr);
  sin_s = (float)sin(w->rotate.yrr);
  sin_t = (float)sin(w->rotate.zrr);
  cos_r = (float)cos(w->rotate.xrr);
  cos_s = (float)cos(w->rotate.yrr);
  cos_t = (float)cos(w->rotate.zrr);

  w->rotate.matrix[0][0] = (cos_s * cos_t);
  w->rotate.matrix[0][1] = -(cos_s * sin_t);
  w->rotate.matrix[0][2] = sin_s;
  w->rotate.matrix[1][0] = ((cos_r * sin_t) + (sin_r * sin_s * cos_t));
  w->rotate.matrix[1][1] = ((cos_r * cos_t) - (sin_r * sin_s * sin_t));
  w->rotate.matrix[1][2] = -(sin_r * cos_s);
  w->rotate.matrix[2][0] = ((sin_r * sin_t) - (cos_r * sin_s * cos_t));
  w->rotate.matrix[2][1] = ((sin_r * cos_t) + (cos_r * sin_s * sin_t));
  w->rotate.matrix[2][2] = (cos_r * cos_s);
  }

/*
 *  rotw_recalc_axes(): reinitialize the axes to original configuration
 */

static void rotw_recalc_axes(RotateWidget cw)
  {
  axes_t *axes = cw->rotate.axes;
  float xx, yy, zz;

  /* origin */
  rotw_project_point(cw, 0, 0, 0, &(axes->ox), &(axes->oy));

  /* z axis */
  rotw_rotate_point(cw, axes->zfx, axes->zfy, axes->zfz, &xx, &yy, &zz);
  axes->zfx = xx, axes->zfy = yy, axes->zfz = zz;
  rotw_project_point(cw, xx, yy, zz, &(axes->zax), &(axes->zay));

  /* y axis */
  rotw_rotate_point(cw, axes->yfx, axes->yfy, axes->yfz, &xx, &yy, &zz);
  axes->yfx = xx, axes->yfy = yy, axes->yfz = zz;
  rotw_project_point(cw, xx, yy, zz, &(axes->yax), &(axes->yay));

  /* x axis */
  rotw_rotate_point(cw, axes->xfx, axes->xfy, axes->xfz, &xx, &yy, &zz);
  axes->xfx = xx, axes->xfy = yy, axes->xfz = zz;
  rotw_project_point(cw, xx, yy, zz, &(axes->xax), &(axes->xay));
  }

/*
 *  rotw_recalc_points(): apply the rotation matrix to the points, the axes,
 *  and the container box
 */

static void rotw_recalc_points(RotateWidget cw)
  {
  XPoint *pt;
  float *xc, *yc, *zc, xx, yy, zz;
  int i;

  cw->rotate.x_scl = (cw->core.width / (cw->rotate.radius * 2));
  cw->rotate.y_scl = (cw->core.height / (cw->rotate.radius * 2));

  /* calculate the points for pointset */

  if(cw->rotate.pointsetf)
    {
    for(i = 0, xc = cw->rotate.lpointset.x, yc = cw->rotate.lpointset.y,
        zc = cw->rotate.lpointset.z, pt = cw->rotate.points;
        i < cw->rotate.lpointset.n; i++, xc++, yc++, zc++, pt++)
      {
      rotw_rotate_point(cw, *xc, *yc, *zc, &xx, &yy, &zz);
      *xc = xx, *yc = yy, *zc = zz;
      rotw_project_point(cw, xx, yy, zz, &(pt->x), &(pt->y));
      }
    }

  /* calculate the points for the lineset */

  if(cw->rotate.linesetf)
    {
    for(i = 0, xc = cw->rotate.llineset.x, yc = cw->rotate.llineset.y,
        zc = cw->rotate.llineset.z, pt = cw->rotate.lines;
        i < cw->rotate.llineset.n; i++, xc++, yc++, zc++, pt++)
      {
      rotw_rotate_point(cw, *xc, *yc, *zc, &xx, &yy, &zz);
      *xc = xx, *yc = yy, *zc = zz;
      rotw_project_point(cw, xx, yy, zz, &(pt->x), &(pt->y));
      }
    }

  /* calculate the points for the container */

  for(i = 0; i < 24; i++)
    {
    rotw_rotate_point(cw, cw->rotate.container.x[i],
		      cw->rotate.container.y[i],
		      cw->rotate.container.z[i], &xx, &yy, &zz);
    cw->rotate.container.x[i] = xx;
    cw->rotate.container.y[i] = yy;
    cw->rotate.container.z[i] = zz;
    }
  }

/*
 *  rotw_init_container(): initialize the values for the container box
 *  line data
 */

static void rotw_init_container(RotateWidget cw)
  {
  static int cxmin[12] = { 0, 1, 2, 7, 8, 9, 10, 15, 16, 17, 18, 19 };
  static int cxmax[12] = { 3, 4, 5, 6, 11, 12, 13, 14, 20, 21, 22, 23 };
  static int cymin[12] = { 0, 1, 2, 3, 4, 5, 6, 7, 16, 18, 20, 22 };
  static int cymax[12] = { 8, 9, 10, 11, 12, 13, 14, 15, 17, 19, 21, 23 };
  static int czmin[12] = { 1, 2, 3, 4, 9, 10, 11, 12, 18, 19, 20, 21 };
  static int czmax[12] = { 0, 5, 6, 7, 8, 13, 14, 15, 16, 17, 22, 23 };
  int i;

  for(i = 0; i < 12; i++)
    {
    cw->rotate.container.x[cxmin[i]] = cw->rotate.minx;
    cw->rotate.container.x[cxmax[i]] = cw->rotate.maxx;
    cw->rotate.container.y[cymin[i]] = cw->rotate.miny;
    cw->rotate.container.y[cymax[i]] = cw->rotate.maxy;
    cw->rotate.container.z[czmin[i]] = cw->rotate.minz;
    cw->rotate.container.z[czmax[i]] = cw->rotate.maxz;
    }
  cw->rotate.container.n = 24;

  /*
     container box is represented as the following line data:

     minx, miny, maxz - minx, miny, minz
     minx, miny, minz - maxx, miny, minz
     maxx, miny, minz - maxx, miny, maxz
     maxx, miny, maxz - minx, miny, maxz

     minx, maxy, maxz - minx, maxy, minz
     minx, maxy, minz - maxx, maxy, minz
     maxx, maxy, minz - maxx, maxy, maxz
     maxx, maxy, maxz - minx, maxy, maxz
     
     minx, miny, maxz - minx, maxy, maxz
     minx, miny, minz - minx, maxy, minz
     maxx, miny, minz - maxx, maxy, minz
     maxx, miny, maxz - maxx, maxy, maxz

     */
  }

/*
 *  rotw_init_axes(): reset axis endpoints to original configuration
 */

static void rotw_init_axes(RotateWidget cw)
  {
  axes_t *axes = cw->rotate.axes;

  axes->xfx = (cw->rotate.radius / AXISLEN);
  axes->yfy = (cw->rotate.radius / AXISLEN);
  axes->zfz = (cw->rotate.radius / AXISLEN);
  axes->xfy = 0;
  axes->xfz = 0;
  axes->yfx = 0;
  axes->yfz = 0;
  axes->zfx = 0;
  axes->zfy = 0;
  }



/*
 *  rotw_vangle(): compute the angle in radians between two 3D vectors
 */

static float rotw_vangle(float ax, float ay, float az, float bx, float by,
			 float bz)
  {
  return((float)acos((double)((ax*bx + ay*by + az*bz) /
			      (float)sqrt((double)((ax*ax + ay*ay + az*az) *
						   (bx*bx + by*by + bz*bz))))));
  }

/* end of source file */
