/****************************************************
program to calculate the relelvant projection
numbers on an obliquely dipping plane.

x1,y1,x2,y2  are the x,y coordinates of the line on the surface
and dip is the dipping angle from the left hand horzontal
positive down when facing in the direction of point X1 towards point X2.
the point x,y,z is the point which is projected on the plane.
RETURN:
aproj is the distance from the point X1 along the line X1-X2.
(negative means the point lies out of the X1-X2 range.  Also
 greater than sqrt(Dx^2 + Dy^2) is beyond the line too).
range is the offset distance from the plane.
dep is the distance from the surface to the new projected point
on the plane.

the function returns 0 if the X1 and X2 points are identical,
which would mean it is not a line.
*/

/* --- feature switches --- */

#define _POSIX_SOURCE 1

/* --- system headers --- */

#include <stdio.h>
#include <math.h>

/* --- local headers --- */

#include "geotouch.h"

/* --- functions --- */
/** FUNC DEF **/double dis2points(float x1, float y1, float x2, float y2)
   {   /* distance between two points  */
   double D;
   D = sqrt( (x2-x1)*(x2-x1) + (y2-y1)*(y2-y1) );
   return(D);
   }


/** FUNC DEF **/void slide_point(float x1, float y1, float x2, float y2, float dis, float *nx, float *ny)
   {    /* find the point which is dis away from point 1 on line between 1 and 2 */
   double dx, dy, D;
   D = dis2points(x1, y1, x2, y2);
   dx = (x2-x1)*dis/D;
   dy = (y2-y1)*dis/D;
   *nx = x1+dx;
   *ny = y1+dy;
   }


/** FUNC DEF **/int gproj(float x1, float y1, float x2, float y2, float dip,
	  float x, float y, float z, float *aproj, float *range, float *dep)
  {
  float dx, dy, aiy, alph, asinz;

  alph = deg2rad(dip);
  dx = x2 - x1;
  dy = y2 - y1;

  aiy = sqrt(dx * dx + dy * dy);
  if(aiy == 0.0)
    {
    *range = 0.0;
    *dep = 0.0;
    return(0);
    }

  *aproj = ((dx * (x - x1) + dy * (y - y1)) / aiy);
  asinz = ((dx * (y - y1) - dy * (x - x1)) / aiy);

  *range = z * cos(alph) - asinz * sin(alph);
  *dep = z * sin(alph) + asinz * cos(alph);

  if((*aproj < 0.0) || (*aproj > aiy) || (*dep < 0.0)) return(2);

  return(1);
  }

/** FUNC DEF **/int norm_vec(float *perp, int len)
  {
  int i;
  double sum = 0.0;

  for(i = 0; i < len; i++) sum += (perp[i] * perp[i]);
  if(sum <= 0.0) return(-1);

  sum = sqrt(sum);
  for(i = 0; i < len; i++) perp[i] /= sum;
  return(1);
  }

/** FUNC DEF **/void get_3roty(float rotmat[3][3], float deg1)
  {
  rotmat[0][0] = cos(deg1);
  rotmat[0][1] = 0.0;
  rotmat[0][2] = -sin(deg1);

  rotmat[1][0] = 0.0;
  rotmat[1][1] = 1.0;
  rotmat[1][2] = 0.0;

  rotmat[2][0] = sin(deg1);
  rotmat[2][1] = 0.0;
  rotmat[2][2] = cos(deg1);
  }

/** FUNC DEF **/void get_3rotx(float rotmat[3][3], float deg1)
  {
  rotmat[0][0] = 1.0;
  rotmat[0][1] = 0.0;
  rotmat[0][2] = 0.0;

  rotmat[1][0] = 0.0;
  rotmat[1][1] = cos(deg1);
  rotmat[1][2] = -sin(deg1);

  rotmat[2][0] = 0.0;
  rotmat[2][1] = sin(deg1);
  rotmat[2][2] = cos(deg1);
  }

/** FUNC DEF **/void get_3rotz(float rotmat[3][3], float deg1)
  {
  rotmat[0][0] = cos(deg1);
  rotmat[0][1] = -sin(deg1);
  rotmat[0][2] = 0.0;

  rotmat[1][0] = sin(deg1);
  rotmat[1][1] = cos(deg1);
  rotmat[1][2] = 0.0;

  rotmat[2][0] = 0.0;
  rotmat[2][1] = 0.0;
  rotmat[2][2] = 1.0;
  }

/** FUNC DEF **/void rot_mat(float rotmat[3][3], float phi, float dip)
  {
  int i, j, k;
  float rotm2[3][3], rotm1[3][3];

  /*
   * get a rotation which first rotates about the
   * z axis phi degrees and then roates dip about the X axis
   * output is a 3X3 matrix which does the rotations of poles
   */

  get_3rotx(rotm2, -dip);
  get_3rotz(rotm1, -phi);
  
  /*
   * combine these two rotations by matrix multiplication
   * rotmat = rotm2 %*% rotm1
   * so that the rotm1 is applied first and then  rotm2 rotation
   */

  for(i = 0; i < 3; i++)
    {
    for(j = 0; j < 3; j++)
      {
      rotmat[i][j] = 0.0;
      for(k = 0; k < 3; k++)
	rotmat[i][j] += rotm2[i][k] * rotm1[k][j];
      }
    }
  }

/** FUNC DEF **/void Rotate_3vector(float rmat[3][3], float *vec, float *v)
  {
  int i, j;

  for(i = 0; i < 3; i++)
    {
    v[i] = 0.0;

    for(j = 0; j < 3; j++)
      v[i] = v[i] + rmat[i][j] * vec[j];
    }
  }

/** FUNC DEF **/void get_rect_vol(float x1, float y1, float x2, float y2, float dip,
		  float mat[3][3])
  {
  float phi, dx = (x2 - x1), dy = (y2 - y1), rdip;

  phi = atan2(dy, dx);
  rdip = deg2rad(dip);
  rot_mat(mat, phi, rdip);
  }

/** FUNC DEF **/ int In_Rect_Vol(float RMAT[3][3], float dx, float dy, float dz, float mindep,
		float maxdep, float mindist, float maxdist, float d1, float d2, float *range,
		float *aproj, float *dep)
  {
  /* this routine is a little warped because of the screwy
     non-right hand rule I am applying in the code.
     Please beware!  
     */

  float v1[3], v2[3], r, d, o;

  v1[0] = dx;
  v1[1] = dy;
  v1[2] = dz;

  Rotate_3vector(RMAT, v1, v2);

  r=v2[0];
  d=v2[2];
  o=v2[1];

  /*     fprintf(stderr,"r=%f dep=%f p=%f md=%f dis=%f d1=%f d2=%f\n", r, o, d,  maxdep,dist, d1,d2  );
   */

  *range = r, *dep = d, *aproj = o;

  return((o >= (mindep - 0.0001)) && (o <= maxdep) && (*range <= maxdist)
	  && (*range >= mindist) && (d <= d1) && (d >= d2));
  }

/** FUNC DEF **/void Dump_3Mat(float r[3][3])
  {
  int i, j;

  for (i = 0; i < 3; i++)
    {
    for (j = 0; j < 3; j++)
      printf("%f ",  r[i][j]);

    putchar('\n');
    }
  }
